<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Test\GraphQl;

use Amasty\RequestQuote\Api\QuoteRepositoryInterface;
use Magento\Framework\Exception\AuthenticationException;
use Magento\GraphQl\Quote\GetMaskedQuoteIdByReservedOrderId;
use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\Quote\Model\MaskedQuoteIdToQuoteIdInterface;
use Magento\Quote\Model\QuoteRepository;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class MoveInQuoteTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'moveInQuote';
    public const CUSTOMER = 'customer_quote@example.com';
    public const CUSTOMER_PASS = 'password';

    /**
     * @var GetMaskedQuoteIdByReservedOrderId
     */
    private $getMaskedQuoteIdByReservedOrderId;

    /**
     * @var CustomerTokenServiceInterface
     */
    private $customerTokenService;

    /**
     * @var MaskedQuoteIdToQuoteIdInterface
     */
    private $maskedQuoteIdToQuoteId;

    /**
     * @var QuoteRepository
     */
    private $quoteRepository;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->getMaskedQuoteIdByReservedOrderId = $objectManager->get(GetMaskedQuoteIdByReservedOrderId::class);
        $this->customerTokenService = $objectManager->get(CustomerTokenServiceInterface::class);
        $this->maskedQuoteIdToQuoteId = $objectManager->get(MaskedQuoteIdToQuoteIdInterface::class);
        $this->quoteRepository = $objectManager->get(QuoteRepository::class);
    }

    /**
     * @group amasty_request_a_quote
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/enable_module.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/set_allowed_groups.php
     *
     * @magentoApiDataFixture Magento/GraphQl/Catalog/_files/simple_product.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/customer.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/create_empty_cart.php
     * @magentoApiDataFixture Magento/GraphQl/Quote/_files/add_simple_product.php
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/active_quote.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/assign_quote_customer_to_quote.php
     */
    public function testMoveInQuote(): void
    {
        $cartMaskId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_quote');
        $requestQuoteMaskId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_order_active');

        $query = $this->getQuery($requestQuoteMaskId, $cartMaskId);
        $response = $this->graphQlMutation($query, [], '', $this->getHeader());

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertTrue($response[self::MAIN_QUERY_KEY]['result']);
        $this->assertEmpty($response[self::MAIN_QUERY_KEY]['warnings']);

        $quoteId = $this->maskedQuoteIdToQuoteId->execute($cartMaskId);

        $this->assertEmpty($this->quoteRepository->get($quoteId)->getItems());
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(string $quoteId, string $cartId): string
    {
        return <<<MUTATION
mutation {
    moveInQuote(
        quoteId:"$quoteId",
        cartId:"$cartId"
    ) {
        result
        warnings
    }
}
MUTATION;
    }

    /**
     * @param string $userName
     * @param string $password
     *
     * @return string[]
     * @throws AuthenticationException
     */
    private function getHeader(
        string $userName = self::CUSTOMER,
        string $password = self::CUSTOMER_PASS
    ): array {
        $customerToken = $this->customerTokenService->createCustomerAccessToken($userName, $password);
        return ['Authorization' => 'Bearer ' . $customerToken];
    }
}
