<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Test\GraphQl;

use Amasty\RequestQuote\Api\QuoteRepositoryInterface;
use Magento\Framework\Exception\AuthenticationException;
use Magento\GraphQl\Quote\GetMaskedQuoteIdByReservedOrderId;
use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\Quote\Model\MaskedQuoteIdToQuoteIdInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class MergeQuotesTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'mergeQuotes';
    public const CUSTOMER = 'customer_quote@example.com';
    public const CUSTOMER_PASS = 'password';

    /**
     * @var GetMaskedQuoteIdByReservedOrderId
     */
    private $getMaskedQuoteIdByReservedOrderId;

    /**
     * @var CustomerTokenServiceInterface
     */
    private $customerTokenService;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->getMaskedQuoteIdByReservedOrderId = $objectManager->get(GetMaskedQuoteIdByReservedOrderId::class);
        $this->customerTokenService = $objectManager->get(CustomerTokenServiceInterface::class);
    }

    /**
     * @group amasty_request_a_quote
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/enable_module.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/set_allowed_groups.php
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/customer.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/second_active_quote/second_active_quote.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/second_active_quote/add_two_products_to_second_cart.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/active_quote.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/assign_quote_customer_to_quote.php
     */
    public function testUpdateQuoteRemarks(): void
    {
        $sourceMaskQuoteId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_order_active_second');
        $destinationMaskQuoteId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_order_active');

        $fields = [
            'amastyQuote' => [
                'status' => 0
            ],
            'items' => [
                [
                    'quantity' => 3,
                    'product' => [
                        'name' => 'Simple Product 10'
                    ]
                ],
                [
                    'quantity' => 1,
                    'product' => [
                        'name' => 'Simple Product 20'
                    ]
                ]
            ]
        ];

        $query = $this->getQuery($sourceMaskQuoteId, $destinationMaskQuoteId);
        $response = $this->graphQlMutation($query, [], '', $this->getHeader());

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_QUERY_KEY], $fields);
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(
        string $sourceQuoteId,
        string $destinationQuoteId
    ): string {
        return <<<MUTATION
mutation {
    mergeQuotes(
        sourceQuoteId: "$sourceQuoteId",
        destinationQuoteId: "$destinationQuoteId"
    ) {
        amastyQuote {
            status
        }
        items{
            quantity
            product {
                name
            }
        }
    }
}
MUTATION;
    }

    /**
     * @param string $userName
     * @param string $password
     *
     * @return string[]
     * @throws AuthenticationException
     */
    private function getHeader(
        string $userName = self::CUSTOMER,
        string $password = self::CUSTOMER_PASS
    ): array {
        $customerToken = $this->customerTokenService->createCustomerAccessToken($userName, $password);
        return ['Authorization' => 'Bearer ' . $customerToken];
    }
}
