<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Test\GraphQl;

use Magento\Framework\Exception\AuthenticationException;
use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class GetCustomerQuoteCartTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'customerQuoteCart';
    public const CUSTOMER = 'customer_quote@example.com';
    public const CUSTOMER_PASS = 'password';

    /**
     * @var CustomerTokenServiceInterface
     */
    private $customerTokenService;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->customerTokenService = $objectManager->get(CustomerTokenServiceInterface::class);
    }

    /**
     * @group amasty_request_a_quote
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/enable_module.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/set_allowed_groups.php
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/customer.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/active_quote.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/assign_quote_customer_to_quote.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/add_customer_note_to_quote.php
     *
     */
    public function testGetCustomerQuoteCart(): void
    {
        $fields = [
            'amastyQuote' => [
                'notes' => [
                    'customer_note' => 'Test graph customer note',
                    'admin_note' => ''
                ],
                'shipping_can_modified' => true,
                'shipping_configured' => false,
                'status' => 0
            ]
        ];

        $query = $this->getQuery();
        $response = $this->graphQlQuery($query, [], '', $this->getHeader());

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_QUERY_KEY], $fields);
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(): string
    {
        return <<<QUERY
query {
    customerQuoteCart {
        amastyQuote {
            notes {
                customer_note
                admin_note
            }
            shipping_can_modified
            shipping_configured
            status
        }
    }
}
QUERY;
    }

    /**
     * @param string $userName
     * @param string $password
     *
     * @return string[]
     * @throws AuthenticationException
     */
    private function getHeader(
        string $userName = self::CUSTOMER,
        string $password = self::CUSTOMER_PASS
    ): array {
        $customerToken = $this->customerTokenService->createCustomerAccessToken($userName, $password);
        return ['Authorization' => 'Bearer ' . $customerToken];
    }
}
