<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Test\GraphQl;

use Magento\TestFramework\Helper\Bootstrap;
use Magento\GraphQl\Quote\GetCartItemOptionsFromUID;
use Magento\TestFramework\TestCase\GraphQlAbstract;
use Magento\GraphQl\Quote\GetCustomOptionsWithUIDForQueryBySku;
use Magento\GraphQl\Quote\GetMaskedQuoteIdByReservedOrderId;

/**
 * Add simple product with custom options to cart using the unified mutation for adding different product types
 */
class AddSimpleProductToCartSingleMutationTest extends GraphQlAbstract
{
    /**
     * @var GetCustomOptionsWithUIDForQueryBySku
     */
    private $getCustomOptionsWithIDV2ForQueryBySku;

    /**
     * @var GetMaskedQuoteIdByReservedOrderId
     */
    private $getMaskedQuoteIdByReservedOrderId;

    /**
     * @var GetCartItemOptionsFromUID
     */
    private $getCartItemOptionsFromUID;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->getMaskedQuoteIdByReservedOrderId = $objectManager->get(GetMaskedQuoteIdByReservedOrderId::class);
        $this->getCartItemOptionsFromUID = $objectManager->get(GetCartItemOptionsFromUID::class);
        $this->getCustomOptionsWithIDV2ForQueryBySku = $objectManager->get(
            GetCustomOptionsWithUIDForQueryBySku::class
        );
    }

    /**
     * Test adding a simple product to the shopping cart with all supported
     * customizable options assigned
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/enable_module.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/set_allowed_groups.php
     *
     * @magentoApiDataFixture Magento/Catalog/_files/product_simple_with_options.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/quote.php
     */
    public function testAddSimpleProductWithOptions(): void
    {
        $maskedQuoteId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_order_111');
        $sku = 'simple';
        $qty = 1;
        $itemOptions = $this->getCustomOptionsWithIDV2ForQueryBySku->execute($sku);
        $decodedItemOptions = $this->getCartItemOptionsFromUID->execute($itemOptions);

        /* The type field is only required for assertions, it should not be present in query */
        foreach ($itemOptions['entered_options'] as &$enteredOption) {
            if (isset($enteredOption['type'])) {
                unset($enteredOption['type']);
            }
        }

        $productOptionsQuery = preg_replace(
            '/"([^"]+)"\s*:\s*/',
            '$1:',
            json_encode($itemOptions)
        );

        $query = $this->getAddToQuoteCartMutation($maskedQuoteId, $qty, $sku, trim($productOptionsQuery, '{}'));
        $response = $this->graphQlMutation($query);

        self::assertArrayHasKey('items', $response['addProductsToQuoteCart']['cart']);
        self::assertCount($qty, $response['addProductsToQuoteCart']['cart']['items']);
        $customizableOptionsOutput =
            $response['addProductsToQuoteCart']['cart']['items'][0]['customizable_options'];

        foreach ($customizableOptionsOutput as $key => $customizableOptionOutput) {
            $customizableOptionOutputValues = [];
            foreach ($customizableOptionOutput['values'] as $customizableOptionOutputValue) {
                $customizableOptionOutputValues[] =  $customizableOptionOutputValue['value'];

                $decodedOptionValue = base64_decode($customizableOptionOutputValue['customizable_option_value_uid']);
                $decodedArray = explode('/', $decodedOptionValue);
                if (count($decodedArray) === 2) {
                    self::assertEquals(
                        base64_encode('custom-option/' . $customizableOptionOutput['id']),
                        $customizableOptionOutputValue['customizable_option_value_uid']
                    );
                } elseif (count($decodedArray) === 3) {
                    self::assertEquals(
                        base64_encode(
                            'custom-option/'
                            . $customizableOptionOutput['id']
                            . '/'
                            . $customizableOptionOutputValue['value']
                        ),
                        $customizableOptionOutputValue['customizable_option_value_uid']
                    );
                } else {
                    self::fail('customizable_option_value_uid ');
                }
            }
            if (count($customizableOptionOutputValues) === 1) {
                $customizableOptionOutputValues = $customizableOptionOutputValues[0];
            }

            self::assertEquals(
                $decodedItemOptions[$customizableOptionOutput['id']],
                $customizableOptionOutputValues
            );

            self::assertEquals(
                base64_encode((string) 'custom-option/' . $customizableOptionOutput['id']),
                $customizableOptionOutput['customizable_option_uid']
            );
        }
    }

    /**
     * @dataProvider wrongSkuDataProvider
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/enable_module.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/set_allowed_groups.php
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/quote.php
     */
    public function testAddProductWithWrongSku(string $sku, string $message): void
    {
        $maskedQuoteId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_order_111');
        $query = $this->getAddToQuoteCartMutation($maskedQuoteId, 1, $sku, '');
        $response = $this->graphQlMutation($query);

        self::assertArrayHasKey('user_errors', $response['addProductsToQuoteCart']);
        self::assertCount(1, $response['addProductsToQuoteCart']['user_errors']);
        self::assertEquals(
            $message,
            $response['addProductsToQuoteCart']['user_errors'][0]['message']
        );
    }

    /**
     * The test covers the case when upon adding available_qty + 1 to the quote cart, the cart is being
     * cleared
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/enable_module.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/set_allowed_groups.php
     *
     * @magentoApiDataFixture Magento/Catalog/_files/product_simple_without_custom_options.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/quote.php
     */
    public function testAddToCartWithQtyPlusOne(): void
    {
        $maskedQuoteId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_order_111');
        $sku = 'simple-2';

        $query = $this->getAddToQuoteCartMutation($maskedQuoteId, 100, $sku, '');
        $response = $this->graphQlMutation($query);

        self::assertEquals(100, $response['addProductsToQuoteCart']['cart']['total_quantity']);

        $query = $this->getAddToQuoteCartMutation($maskedQuoteId, 1, $sku, '');
        $response = $this->graphQlMutation($query);

        self::assertArrayHasKey('user_errors', $response['addProductsToQuoteCart']);
        self::assertEquals(
            'The requested qty is not available',
            $response['addProductsToQuoteCart']['user_errors'][0]['message']
        );
        self::assertEquals(100, $response['addProductsToQuoteCart']['cart']['total_quantity']);
    }

    /**
     * @dataProvider wrongQuantityDataProvider
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/enable_module.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/set_allowed_groups.php
     *
     * @magentoApiDataFixture Magento/Catalog/_files/product_simple_without_custom_options.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/quote.php
     */
    public function testAddProductWithWrongQuantity(int $quantity, string $message): void
    {
        $maskedQuoteId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_order_111');
        $sku = 'simple-2';
        $query = $this->getAddToQuoteCartMutation($maskedQuoteId, $quantity, $sku, '');
        $response = $this->graphQlMutation($query);

        self::assertArrayHasKey('user_errors', $response['addProductsToQuoteCart']);
        self::assertCount(1, $response['addProductsToQuoteCart']['user_errors']);
        self::assertEquals(
            $message,
            $response['addProductsToQuoteCart']['user_errors'][0]['message']
        );
    }

    /**
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/enable_module.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/set_allowed_groups.php
     *
     * @magentoApiDataFixture Magento/Catalog/_files/products_with_websites_and_stores.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/quote.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/quote_not_default_website.php
     * @dataProvider addProductNotAssignedToWebsiteDataProvider
     */
    public function testAddProductNotAssignedToWebsite(string $reservedQuoteId, string $sku, array $headerMap): void
    {
        $maskedQuoteId = $this->getMaskedQuoteIdByReservedOrderId->execute($reservedQuoteId);
        $query = $this->getAddToQuoteCartMutation($maskedQuoteId, 1, $sku);
        $response = $this->graphQlMutation($query, [], '', $headerMap);
        self::assertEmpty($response['addProductsToQuoteCart']['cart']['items']);
        self::assertArrayHasKey('user_errors', $response['addProductsToQuoteCart']);
        self::assertCount(1, $response['addProductsToQuoteCart']['user_errors']);
        self::assertStringContainsString($sku, $response['addProductsToQuoteCart']['user_errors'][0]['message']);
        self::assertEquals('PRODUCT_NOT_FOUND', $response['addProductsToQuoteCart']['user_errors'][0]['code']);
    }

    public function addProductNotAssignedToWebsiteDataProvider(): array
    {
        return [
            ['test_order_111', 'simple-2', []],
            ['test_order_111', 'simple-2', ['Store' => 'default']],
            ['test_order_222', 'simple-1', ['Store' => 'fixture_second_store']],
        ];
    }

    public function wrongSkuDataProvider(): array
    {
        return [
            'Non-existent SKU' => [
                'non-existent',
                'Could not find a product with SKU "non-existent"'
            ],
            'Empty SKU' => [
                '',
                'Could not find a product with SKU ""'
            ]
        ];
    }

    public function wrongQuantityDataProvider(): array
    {
        return [
            'More quantity than in stock' => [
                101,
                'The requested qty is not available'
            ],
            'Quantity equals zero' => [
                0,
                'The product quantity should be greater than 0'
            ]
        ];
    }

    /**
     * Returns GraphQl query string
     */
    private function getAddToQuoteCartMutation(
        string $maskedQuoteId,
        int $qty,
        string $sku,
        string $customizableOptions = ''
    ): string {
        return <<<MUTATION
mutation {
    addProductsToQuoteCart(
        quoteId: "{$maskedQuoteId}",
        quoteItems: [
            {
                sku: "{$sku}"
                quantity: {$qty}
                {$customizableOptions}
            }
        ]
    ) {
        cart {
            total_quantity
            items {
                quantity
                ... on SimpleCartItem {
                    customizable_options {
                        label
                        id
                        customizable_option_uid
                          values {
                            value
                            customizable_option_value_uid
                            id
                        }
                    }
                }
            }
        },
        user_errors {
            code
            message
        }
    }
}
MUTATION;
    }
}
