<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver;

use Amasty\RequestAQuoteGraphql\Model\Di\Wrapper as DiWrapper;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\GetQuoteForUser;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\IsEnabledForCustomer as IsQuoteCartEnabledProvider;
use Amasty\RequestAQuoteGraphql\Model\QuoteItem\DataProvider\UpdateQuoteItems as UpdateQuoteItemsProvider;
use Amasty\RequestAQuoteGraphql\Model\Resolver\UpdateQuoteItems\Input\PrepareAssociativeItems;
use Amasty\RequestAQuoteGraphql\Model\Resolver\UpdateQuoteItems\Input\ResolveNestedCartItemsInput;
use Amasty\RequestQuote\Api\QuoteRepositoryInterface;
use Amasty\RequestQuote\Helper\Cart as CartHelper;
use Amasty\RequestQuote\Model\Quote\Frontend\UpdateQuoteItems as UpdateQuoteItemsProcessor;
use Amasty\RequestQuote\Model\Quote\Frontend\UpdateQuoteItems\UpdateRequestedPrice;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\Resolver\ArgumentsProcessorInterface;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;

class UpdateQuoteItems implements ResolverInterface
{
    public const INPUT_CODE = 'input';
    public const CART_ID_CODE = 'cart_id';
    public const CART_ITEMS_CODE = 'cart_items';
    public const CART_ITEM_ID_CODE = 'cart_item_id';
    public const CUSTOM_ITEM_INPUT_CODE = 'cartItemUpdateInput';
    public const REMARKS_CODE = 'remarks';

    /**
     * @var ResolveNestedCartItemsInput
     */
    private $resolveNestedCartItemsInput;

    /**
     * @var ArgumentsProcessorInterface|DiWrapper
     */
    private $argsSelection;

    /**
     * @var PrepareAssociativeItems
     */
    private $prepareAssociativeItems;

    /**
     * @var GetQuoteForUser
     */
    private $getQuoteForUser;

    /**
     * @var UpdateQuoteItemsProcessor
     */
    private $updateQuoteItemsProcessor;

    /**
     * @var UpdateRequestedPrice
     */
    private $updateRequestedPrice;

    /**
     * @var UpdateQuoteItemsProvider
     */
    private $updateQuoteItemsProvider;

    /**
     * @var QuoteRepositoryInterface
     */
    private $quoteRepository;

    /**
     * @var CartHelper
     */
    private $cartHelper;

    /**
     * @var IsQuoteCartEnabledProvider
     */
    private $isQuoteCartEnabled;

    public function __construct(
        ResolveNestedCartItemsInput $resolveNestedCartItemsInput,
        DiWrapper $argsSelection,
        PrepareAssociativeItems $prepareAssociativeItems,
        GetQuoteForUser $getQuoteForUser,
        UpdateQuoteItemsProcessor $updateQuoteItemsProcessor,
        UpdateRequestedPrice $updateRequestedPrice,
        UpdateQuoteItemsProvider $updateQuoteItemsProvider,
        QuoteRepositoryInterface $quoteRepository,
        CartHelper $cartHelper,
        IsQuoteCartEnabledProvider $isQuoteCartEnabled
    ) {
        $this->resolveNestedCartItemsInput = $resolveNestedCartItemsInput;
        $this->argsSelection = $argsSelection;
        $this->prepareAssociativeItems = $prepareAssociativeItems;
        $this->getQuoteForUser = $getQuoteForUser;
        $this->updateQuoteItemsProcessor = $updateQuoteItemsProcessor;
        $this->updateRequestedPrice = $updateRequestedPrice;
        $this->updateQuoteItemsProvider = $updateQuoteItemsProvider;
        $this->quoteRepository = $quoteRepository;
        $this->cartHelper = $cartHelper;
        $this->isQuoteCartEnabled = $isQuoteCartEnabled;
    }

    /**
     * @param Field $field
     * @param $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlInputException
     * @throws GraphQlAuthorizationException
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$this->isQuoteCartEnabled->execute($context->getExtensionAttributes()->getCustomerGroupId())) {
            throw new GraphQlAuthorizationException(__('Quote Cart has been disabled.'));
        }

        if (empty($args[self::INPUT_CODE][self::CART_ID_CODE])) {
            throw new GraphQlInputException(__('Required parameter "cart_id" is missing.'));
        }

        if (empty($args[self::INPUT_CODE][self::CART_ITEMS_CODE])
            || !is_array($args[self::INPUT_CODE][self::CART_ITEMS_CODE])
        ) {
            throw new GraphQlInputException(__('Required parameter "cart_items" is missing.'));
        }

        $args = $this->resolveNestedCartItemsInput->execute($args);
        $processedArgs = $this->argsSelection->process($info->fieldName, $args) ?: $args;
        $processedArgs = $this->prepareAssociativeItems->execute($processedArgs);

        $maskedCartId = $processedArgs[self::INPUT_CODE][self::CART_ID_CODE];
        $quoteItemsData = $processedArgs[self::INPUT_CODE][self::CART_ITEMS_CODE];
        $storeId = (int)$context->getExtensionAttributes()->getStore()->getId();
        $quote = $this->getQuoteForUser->execute($maskedCartId, $context->getUserId(), $storeId);

        if (!empty($processedArgs[self::INPUT_CODE][self::REMARKS_CODE])
            && trim($processedArgs[self::INPUT_CODE][self::REMARKS_CODE])
        ) {
            $quote->setRemarks(
                $this->cartHelper->prepareCustomerNoteForSave($processedArgs[self::INPUT_CODE][self::REMARKS_CODE])
            );
        }
        try {
            $quoteItems = $this->updateQuoteItemsProcessor->execute($quote, $quoteItemsData);
            $this->updateQuoteItemsProvider->execute($quote, $quoteItemsData);
            $quote->collectTotals();
            $this->updateRequestedPrice->execute($quoteItems);
            $this->quoteRepository->save($quote);
        } catch (NoSuchEntityException $e) {
            throw new GraphQlNoSuchEntityException(__($e->getMessage()), $e);
        } catch (LocalizedException $e) {
            throw new GraphQlInputException(__($e->getMessage()), $e);
        }

        return [
            'cart' => [
                'model' => $quote,
            ]
        ];
    }
}
