<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver;

use Amasty\RequestAQuoteGraphql\Model\Di\Wrapper as DiWrapper;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\GetQuoteForUser;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\IsEnabledForCustomer as IsQuoteCartEnabledProvider;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\SubmitQuote\GenerateCustomerToken;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\SubmitQuote\SubmitQuote as SubmitQuoteService;
use Amasty\RequestAQuoteGraphql\Model\QuoteItem\DataProvider\UpdateQuoteItems as UpdateQuoteItemsProvider;
use Amasty\RequestAQuoteGraphql\Model\Resolver\UpdateQuoteItems\Input\PrepareAssociativeItems;
use Amasty\RequestAQuoteGraphql\Model\Resolver\UpdateQuoteItems\Input\ResolveNestedCartItemsInput;
use Amasty\RequestQuote\Api\Data\QuoteInterface;
use Amasty\RequestQuote\Helper\Cart as CartHelper;
use Amasty\RequestQuote\Model\ConfigProvider;
use Amasty\RequestQuote\Model\Quote\Frontend\UpdateQuoteItems as UpdateQuoteItemsProcessor;
use Amasty\RequestQuote\Model\Quote\Frontend\UpdateQuoteItems\UpdateRequestedPrice;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\Resolver\ArgumentsProcessorInterface;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Validator\Exception as ValidatorException;
use Magento\GraphQl\Model\Query\ContextInterface;

class SubmitQuote implements ResolverInterface
{
    private const INPUT_CODE = 'input';
    private const UPDATE_QUOTE_ITEMS_CODE = 'updateQuoteItemsInput';
    private const CUSTOMER_INPUT_CODE = 'customerInput';

    /**
     * @var ResolveNestedCartItemsInput
     */
    private $resolveNestedCartItemsInput;

    /**
     * @var ArgumentsProcessorInterface|DiWrapper
     */
    private $argsSelection;

    /**
     * @var PrepareAssociativeItems
     */
    private $prepareAssociativeItems;

    /**
     * @var GetQuoteForUser
     */
    private $getQuoteForUser;

    /**
     * @var CartHelper
     */
    private $cartHelper;

    /**
     * @var UpdateQuoteItemsProcessor
     */
    private $updateQuoteItemsProcessor;

    /**
     * @var UpdateQuoteItemsProvider
     */
    private $updateQuoteItemsProvider;

    /**
     * @var GenerateCustomerToken
     */
    private $generateCustomerToken;

    /**
     * @var SubmitQuoteService
     */
    private $submitQuoteService;

    /**
     * @var UpdateRequestedPrice
     */
    private $updateRequestedPrice;

    /**
     * @var IsQuoteCartEnabledProvider
     */
    private $isQuoteCartEnabled;

    /**
     * @var ConfigProvider|null
     */
    private $configProvider;

    public function __construct(
        ResolveNestedCartItemsInput $resolveNestedCartItemsInput,
        DiWrapper $argsSelection,
        PrepareAssociativeItems $prepareAssociativeItems,
        GetQuoteForUser $getQuoteForUser,
        CartHelper $cartHelper,
        UpdateQuoteItemsProcessor $updateQuoteItemsProcessor,
        UpdateQuoteItemsProvider $updateQuoteItemsProvider,
        GenerateCustomerToken $generateCustomerToken,
        SubmitQuoteService $submitQuoteService,
        UpdateRequestedPrice $updateRequestedPrice,
        IsQuoteCartEnabledProvider $isQuoteCartEnabled,
        ?ConfigProvider $configProvider = null
    ) {
        $this->resolveNestedCartItemsInput = $resolveNestedCartItemsInput;
        $this->argsSelection = $argsSelection;
        $this->prepareAssociativeItems = $prepareAssociativeItems;
        $this->getQuoteForUser = $getQuoteForUser;
        $this->cartHelper = $cartHelper;
        $this->updateQuoteItemsProcessor = $updateQuoteItemsProcessor;
        $this->updateQuoteItemsProvider = $updateQuoteItemsProvider;
        $this->generateCustomerToken = $generateCustomerToken;
        $this->submitQuoteService = $submitQuoteService;
        $this->updateRequestedPrice = $updateRequestedPrice;
        $this->isQuoteCartEnabled = $isQuoteCartEnabled;
        $this->configProvider = $configProvider ?? ObjectManager::getInstance()->get(ConfigProvider::class);
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlNoSuchEntityException
     * @throws GraphQlInputException
     * @throws GraphQlAuthorizationException
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$this->isQuoteCartEnabled->execute($context->getExtensionAttributes()->getCustomerGroupId())) {
            throw new GraphQlAuthorizationException(__('Quote Cart has been disabled.'));
        }

        if (empty($args[self::INPUT_CODE][self::UPDATE_QUOTE_ITEMS_CODE])) {
            throw new GraphQlInputException(__(
                'Required parameter "%1" is missing.',
                self::UPDATE_QUOTE_ITEMS_CODE
            ));
        }

        $updateQuoteItemsData = $args[self::INPUT_CODE][self::UPDATE_QUOTE_ITEMS_CODE];

        if (empty($updateQuoteItemsData[UpdateQuoteItems::CART_ID_CODE])) {
            throw new GraphQlInputException(__(
                'Required parameter "%1" is missing.',
                UpdateQuoteItems::CART_ID_CODE
            ));
        }
        $maskedCartId = $updateQuoteItemsData[UpdateQuoteItems::CART_ID_CODE];
        $quoteItemsData = $this->getQuoteItemsData($info, $args);
        $storeId = (int) $context->getExtensionAttributes()->getStore()->getId();
        $quote = $this->getQuoteForUser->execute($maskedCartId, $context->getUserId(), $storeId);

        if (!$quote->getAllVisibleItems()) {
            throw new GraphQlInputException(__('Empty Quote Cart can not be submitted.'));
        }

        if (!empty($updateQuoteItemsData[UpdateQuoteItems::REMARKS_CODE])
            && trim($updateQuoteItemsData[UpdateQuoteItems::REMARKS_CODE])
        ) {
            $quote->setRemarks($this->cartHelper->prepareCustomerNoteForSave(
                $updateQuoteItemsData[UpdateQuoteItems::REMARKS_CODE]
            ));
        }
        try {
            $quoteItems = $this->updateQuoteItemsProcessor->execute($quote, $quoteItemsData);
            $this->updateQuoteItemsProvider->execute($quote, $quoteItemsData);
            if ($context->getExtensionAttributes()->getIsCustomer() === false) {
                if ($this->configProvider->isAllowedCreateAccountForGuest($storeId)) {
                    $customerToken = $this->generateCustomerToken->execute(
                        $args[self::INPUT_CODE][self::CUSTOMER_INPUT_CODE],
                        $context->getExtensionAttributes()->getStore()
                    );
                    $quote->setCustomer($customerToken->getCustomer());
                } else {
                    $this->updateQuoteForGuest($quote, $args[self::INPUT_CODE][self::CUSTOMER_INPUT_CODE]);
                }
            }
            $quote->collectTotals();
            $this->updateRequestedPrice->execute($quoteItems);

            $this->submitQuoteService->execute($quote);
        } catch (NoSuchEntityException $e) {
            throw new GraphQlNoSuchEntityException(__($e->getMessage()), $e);
        } catch (LocalizedException $e) {
            throw new GraphQlInputException(__($e->getMessage()), $e);
        }

        return [
            'cart' => [
                'model' => $quote
            ],
            'token' => isset($customerToken) ? $customerToken->getToken() : null
        ];
    }

    private function getQuoteItemsData(ResolveInfo $info, array $args): array
    {
        $argsForUpdateItems = [UpdateQuoteItems::INPUT_CODE => $args[self::INPUT_CODE][self::UPDATE_QUOTE_ITEMS_CODE]];
        $argsForUpdateItems = $this->resolveNestedCartItemsInput->execute($argsForUpdateItems);
        $argsForUpdateItems = $this->argsSelection->process($info->fieldName, $argsForUpdateItems)
            ?: $argsForUpdateItems;
        $argsForUpdateItems = $this->prepareAssociativeItems->execute($argsForUpdateItems);

        return $argsForUpdateItems[UpdateQuoteItems::INPUT_CODE][UpdateQuoteItems::CART_ITEMS_CODE];
    }

    private function updateQuoteForGuest(QuoteInterface $quote, array $customerData): void
    {
        $messages = [];

        $email = $customerData['email'] ?? null;
        if (!$email) {
            $messages['email'] = [__('"Email" is a required value.')];
        }

        $firstName = $customerData['firstname'] ?? null;
        if (!$firstName) {
            $messages['firstname'] = [__('"First Name" is a required value.')];
        }

        $lastName = $customerData['lastname'] ?? null;
        if (!$lastName) {
            $messages['lastname'] = [__('"Last Name" is a required value.')];
        }

        if ($messages) {
            throw new ValidatorException(null, null, $messages);
        }

        $quote->setCustomerLastname($lastName);
        $quote->setCustomerFirstname($firstName);
        $quote->setCustomerEmail($email);
        $quote->setCustomerIsGuest(true);
    }
}
