<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver;

use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\GraphQl\Model\Query\ContextInterface;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Model\Quote\Item as QuoteItem;
use Magento\QuoteGraphQl\Model\Cart\GetCartForUser;

class RemoveQuoteFromCart implements ResolverInterface
{
    /**
     * @var GetCartForUser
     */
    private $getCartForUser;

    /**
     * @var CartRepositoryInterface
     */
    private $cartRepository;

    public function __construct(GetCartForUser $getCartForUser, CartRepositoryInterface $cartRepository)
    {
        $this->getCartForUser = $getCartForUser;
        $this->cartRepository = $cartRepository;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlAuthorizationException
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$context->getExtensionAttributes()->getIsCustomer()) {
            throw new GraphQlAuthorizationException(__('The current customer isn\'t authorized.'));
        }

        $quote = $this->getCartForUser->execute(
            $args['cartId'],
            $context->getUserId(),
            (int) $context->getExtensionAttributes()->getStore()->getId()
        );
        foreach ($quote->getAllVisibleItems() as $quoteItem) {
            if ($this->isAmastyQuoteItem($quoteItem)) {
                $quoteItem->removeOption('amasty_quote_price');
                $quote->deleteItem($quoteItem);
            }
        }

        $this->cartRepository->save($quote);

        return [
            'model' => $quote
        ];
    }

    private function isAmastyQuoteItem(QuoteItem $quoteItem): bool
    {
        return (bool) $quoteItem->getOptionByCode('amasty_quote_price');
    }
}
