<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver;

use Amasty\RequestAQuoteGraphql\Model\Di\Wrapper as DiWrapper;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\GetQuoteForUser;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\IsEnabledForCustomer as IsQuoteCartEnabledProvider;
use Amasty\RequestQuote\Api\QuoteItemRepositoryInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\Resolver\ArgumentsProcessorInterface;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\GraphQl\Model\Query\Context;

class RemoveItemFromQuoteCart implements ResolverInterface
{
    /**
     * @var GetQuoteForUser
     */
    private $getQuoteForUser;

    /**
     * @var QuoteItemRepositoryInterface
     */
    private $quoteItemRepository;

    /**
     * @var DiWrapper|ArgumentsProcessorInterface
     */
    private $argsSelection;

    /**
     * @var IsQuoteCartEnabledProvider
     */
    private $isQuoteCartEnabled;

    public function __construct(
        GetQuoteForUser $getQuoteForUser,
        QuoteItemRepositoryInterface $quoteItemRepository,
        DiWrapper $argsSelection,
        IsQuoteCartEnabledProvider $isQuoteCartEnabled
    ) {
        $this->getQuoteForUser = $getQuoteForUser;
        $this->quoteItemRepository = $quoteItemRepository;
        $this->argsSelection = $argsSelection;
        $this->isQuoteCartEnabled = $isQuoteCartEnabled;
    }

    /**
     * @param Field $field
     * @param Context $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws LocalizedException
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$this->isQuoteCartEnabled->execute($context->getExtensionAttributes()->getCustomerGroupId())) {
            throw new GraphQlAuthorizationException(__('Quote Cart has been disabled.'));
        }

        $processedArgs = $this->argsSelection->process($info->fieldName, $args) ?: $args;
        if (empty($processedArgs['input']['cart_id'])) {
            throw new GraphQlInputException(__('Required parameter "cart_id" is missing.'));
        }

        $cart = $this->getQuoteForUser->execute(
            $processedArgs['input']['cart_id'],
            $context->getUserId(),
            (int) $context->getExtensionAttributes()->getStore()->getId()
        );

        if (empty($processedArgs['input']['cart_item_id'])) {
            throw new GraphQlInputException(__('Required parameter "cart_item_id" is missing.'));
        }
        $itemId = (int) $processedArgs['input']['cart_item_id'];

        try {
            $this->quoteItemRepository->deleteById((int) $cart->getId(), $itemId);
        } catch (NoSuchEntityException $e) {
            throw new GraphQlNoSuchEntityException(__('The cart doesn\'t contain the item'));
        } catch (LocalizedException $e) {
            throw new GraphQlInputException(__($e->getMessage()), $e);
        }

        return [
            'cart' => [
                'model' => $cart
            ]
        ];
    }
}
