<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver;

use Amasty\RequestAQuoteGraphql\Model\QuoteCart\GetQuoteForUser;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\IsEnabledForCustomer as IsQuoteCartEnabledProvider;
use Amasty\RequestQuote\Api\QuoteRepositoryInterface as AmastyQuoteRepository;
use Amasty\RequestQuote\Model\Quote\Frontend\GetAmastyQuote;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Phrase;
use Magento\GraphQl\Model\Query\ContextInterface;
use Magento\Quote\Api\CartRepositoryInterface as MagentoQuoteRepository;
use Magento\Quote\Model\MaskedQuoteIdToQuoteIdInterface;
use Magento\QuoteGraphQl\Model\Cart\GetCartForUser;

class MoveInQuote implements ResolverInterface
{
    /**
     * @var IsQuoteCartEnabledProvider
     */
    private $isQuoteCartEnabled;

    /**
     * @var GetCartForUser
     */
    private $getCartForUser;

    /**
     * @var GetAmastyQuote
     */
    private $getAmastyQuote;

    /**
     * @var GetQuoteForUser
     */
    private $getQuoteForUser;

    /**
     * @var MaskedQuoteIdToQuoteIdInterface
     */
    private $maskedQuoteIdToQuoteId;

    /**
     * @var AmastyQuoteRepository
     */
    private $amastyQuoteRepository;

    /**
     * @var MagentoQuoteRepository
     */
    private $magentoQuoteRepository;

    public function __construct(
        IsQuoteCartEnabledProvider $isQuoteCartEnabled,
        GetCartForUser $getCartForUser,
        GetAmastyQuote $getAmastyQuote,
        GetQuoteForUser $getQuoteForUser,
        MaskedQuoteIdToQuoteIdInterface $maskedQuoteIdToQuoteId,
        AmastyQuoteRepository $amastyQuoteRepository,
        MagentoQuoteRepository $magentoQuoteRepository
    ) {
        $this->isQuoteCartEnabled = $isQuoteCartEnabled;
        $this->getCartForUser = $getCartForUser;
        $this->getAmastyQuote = $getAmastyQuote;
        $this->getQuoteForUser = $getQuoteForUser;
        $this->maskedQuoteIdToQuoteId = $maskedQuoteIdToQuoteId;
        $this->amastyQuoteRepository = $amastyQuoteRepository;
        $this->magentoQuoteRepository = $magentoQuoteRepository;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlAuthorizationException
     * @throws GraphQlInputException
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$this->isQuoteCartEnabled->execute($context->getExtensionAttributes()->getCustomerGroupId())) {
            throw new GraphQlAuthorizationException(__('Quote Cart has been disabled.'));
        }

        if (empty($args['quoteId'])) {
            throw new GraphQlInputException(__('Required parameter "quoteId" is missing'));
        }
        if (empty($args['cartId'])) {
            throw new GraphQlInputException(__('Required parameter "cartId" is missing'));
        }
        $cart = $this->getCartForUser->execute(
            $args['cartId'],
            $context->getUserId(),
            (int) $context->getExtensionAttributes()->getStore()->getId()
        );
        if ($this->getAmastyQuote->execute($cart)) {
            throw new GraphQlAuthorizationException(__('This is approved quote.'));
        }

        $quote = $this->getQuoteForUser->execute(
            $args['quoteId'],
            $context->getUserId(),
            (int) $context->getExtensionAttributes()->getStore()->getId()
        );

        $mergeResult = $quote->advancedMerge($cart, true, true);

        $this->amastyQuoteRepository->save($quote);
        $this->magentoQuoteRepository->save($cart);

        return [
            'result' => $mergeResult->getResult(),
            'warnings' => array_map(function (Phrase $phrase) {
                return $phrase->render();
            }, $mergeResult->getWarnings())
        ];
    }
}
