<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver;

use Amasty\RequestAQuoteGraphql\Model\QuoteCart\IsEnabled as IsQuoteCartEnabledProvider;
use Amasty\RequestQuote\Api\Data\QuoteInterface;
use Amasty\RequestQuote\Api\QuoteRepositoryInterface as AmastyQuoteRepository;
use Amasty\RequestQuote\Model\Quote\Frontend\GetAmastyQuote;
use Amasty\RequestQuote\Model\Quote\Move\MergeQuotes;
use Amasty\RequestQuote\Model\Source\Status;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\GraphQl\Model\Query\ContextInterface;
use Magento\Quote\Api\CartRepositoryInterface as MagentoQuoteRepository;
use Magento\Quote\Model\MaskedQuoteIdToQuoteIdInterface;
use Magento\QuoteGraphQl\Model\Cart\GetCartForUser;

class MoveInCart implements ResolverInterface
{
    /**
     * @var IsQuoteCartEnabledProvider
     */
    private $isQuoteCartEnabled;

    /**
     * @var GetCartForUser
     */
    private $getCartForUser;

    /**
     * @var GetAmastyQuote
     */
    private $getAmastyQuote;

    /**
     * @var AmastyQuoteRepository
     */
    private $amastyQuoteRepository;

    /**
     * @var MagentoQuoteRepository
     */
    private $magentoQuoteRepository;

    /**
     * @var MergeQuotes
     */
    private $mergeQuotes;

    /**
     * @var MaskedQuoteIdToQuoteIdInterface
     */
    private $maskedQuoteIdToQuoteId;

    public function __construct(
        IsQuoteCartEnabledProvider $isQuoteCartEnabled,
        GetCartForUser $getCartForUser,
        GetAmastyQuote $getAmastyQuote,
        AmastyQuoteRepository $amastyQuoteRepository,
        MagentoQuoteRepository $magentoQuoteRepository,
        MergeQuotes $mergeQuotes,
        MaskedQuoteIdToQuoteIdInterface $maskedQuoteIdToQuoteId
    ) {
        $this->isQuoteCartEnabled = $isQuoteCartEnabled;
        $this->getCartForUser = $getCartForUser;
        $this->getAmastyQuote = $getAmastyQuote;
        $this->amastyQuoteRepository = $amastyQuoteRepository;
        $this->magentoQuoteRepository = $magentoQuoteRepository;
        $this->mergeQuotes = $mergeQuotes;
        $this->maskedQuoteIdToQuoteId = $maskedQuoteIdToQuoteId;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return bool
     * @throws GraphQlAuthorizationException
     * @throws GraphQlInputException
     * @throws GraphQlNoSuchEntityException
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$this->isQuoteCartEnabled->execute()) {
            throw new GraphQlAuthorizationException(__('Quote Cart has been disabled.'));
        }

        if (false === $context->getExtensionAttributes()->getIsCustomer()) {
            throw new GraphQlAuthorizationException(__('The request is allowed for logged in customer'));
        }

        if (empty($args['quoteUid'])) {
            throw new GraphQlInputException(__('Required parameter "quoteUid" is missing'));
        }
        if (empty($args['cartId'])) {
            throw new GraphQlInputException(__('Required parameter "cartId" is missing'));
        }

        // phpcs:ignore Magento2.Functions.DiscouragedFunction
        $quoteId = (int) base64_decode($args['quoteUid'], true);
        if (!$quoteId) {
            throw new GraphQlInputException(__('Value of uid "%1" is incorrect.', $args['quoteUid']));
        }

        $cart = $this->getCartForUser->execute(
            $args['cartId'],
            $context->getUserId(),
            (int) $context->getExtensionAttributes()->getStore()->getId()
        );

        if ($this->getAmastyQuote->execute($cart)) {
            throw new GraphQlAuthorizationException(
                __(
                    'It is possible to process one Quote at a time.
                        You have already added Quote in your cart. Please proceed to checkout.'
                )
            );
        }

        try {
            /** @var QuoteInterface $approvedQuote */
            $approvedQuote = $this->amastyQuoteRepository->get($quoteId);
        } catch (NoSuchEntityException $e) {
            throw new GraphQlNoSuchEntityException(
                __('Could not find a quote with UID "%quote_uid"', ['quote_uid' => $args['quoteUid']])
            );
        }

        if ((int) $approvedQuote->getCustomerId() !== $context->getUserId()) {
            throw new GraphQlAuthorizationException(
                __(
                    'The current user cannot perform operations on quote "%quote_uid"',
                    ['quote_uid' => $args['quoteUid']]
                )
            );
        }

        if ($approvedQuote->getStatus() === Status::APPROVED) {
            $this->mergeQuotes->execute($cart, $approvedQuote);
            $this->magentoQuoteRepository->save($cart);
        } else {
            throw new GraphQlAuthorizationException(__('Quote with UID %1 not approved', $args['quoteUid']));
        }

        return true;
    }
}
