<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver;

use Amasty\RequestAQuoteGraphql\Model\QuoteCart\CustomerQuoteResolver;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\GetQuoteForUser;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\IsEnabledForCustomer as IsQuoteCartEnabledProvider;
use Amasty\RequestQuote\Api\QuoteRepositoryInterface;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\GraphQl\Model\Query\ContextInterface;

class MergeQuotes implements ResolverInterface
{
    private const SOURCE_QUOTE_ID_CODE = 'sourceQuoteId';
    private const DESTINATION_QUOTE_ID_CODE = 'destinationQuoteId';

    /**
     * @var CustomerQuoteResolver
     */
    private $customerQuoteResolver;

    /**
     * @var GetQuoteForUser
     */
    private $getQuoteForUser;

    /**
     * @var QuoteRepositoryInterface
     */
    private $quoteRepository;

    /**
     * @var IsQuoteCartEnabledProvider
     */
    private $isQuoteCartEnabled;

    public function __construct(
        CustomerQuoteResolver $customerQuoteResolver,
        GetQuoteForUser $getQuoteForUser,
        QuoteRepositoryInterface $quoteRepository,
        IsQuoteCartEnabledProvider $isQuoteCartEnabled
    ) {
        $this->customerQuoteResolver = $customerQuoteResolver;
        $this->getQuoteForUser = $getQuoteForUser;
        $this->quoteRepository = $quoteRepository;
        $this->isQuoteCartEnabled = $isQuoteCartEnabled;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlAuthorizationException
     * @throws GraphQlNoSuchEntityException
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$this->isQuoteCartEnabled->execute($context->getExtensionAttributes()->getCustomerGroupId())) {
            throw new GraphQlAuthorizationException(__('Quote Cart has been disabled.'));
        }
        if ($context->getExtensionAttributes()->getIsCustomer() === false) {
            throw new GraphQlAuthorizationException(__('The current customer isn\'t authorized.'));
        }

        $currentUserId = $context->getUserId();
        $storeId = (int) $context->getExtensionAttributes()->getStore()->getId();

        if (empty($args[self::DESTINATION_QUOTE_ID_CODE])) {
            try {
                $customerQuote = $this->customerQuoteResolver->resolve($currentUserId);
            } catch (CouldNotSaveException $exception) {
                throw new GraphQlNoSuchEntityException(
                    __('Could not create empty cart for customer'),
                    $exception
                );
            }
        } else {
            $customerQuote = $this->getQuoteForUser->execute(
                $args[self::DESTINATION_QUOTE_ID_CODE],
                $currentUserId,
                $storeId
            );
        }
        $guestQuote = $this->getQuoteForUser->execute($args[self::SOURCE_QUOTE_ID_CODE], null, $storeId);

        $customerQuote->merge($guestQuote);

        $this->quoteRepository->save($customerQuote);
        $this->quoteRepository->delete($guestQuote);

        return [
            'model' => $customerQuote
        ];
    }
}
