<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver\CustomerQuotes\Query;

use Magento\Framework\Api\Filter;
use Magento\Framework\Api\FilterBuilder;
use Magento\Framework\Api\Search\FilterGroup;
use Magento\Framework\Api\Search\FilterGroupBuilder;
use Magento\Framework\Exception\InputException;

/**
 * Quote filter allows to filter collection using 'increment_id' as quote number, from the search criteria.
 */
class QuoteFilter
{
    /**
     * Translator field from graphql to collection field
     *
     * @var string[]
     */
    private $fieldTranslatorArray = [
        'number' => 'increment_id',
    ];

    /**
     * @var FilterBuilder
     */
    private $filterBuilder;

    /**
     * @var FilterGroupBuilder
     */
    private $filterGroupBuilder;

    public function __construct(
        FilterBuilder $filterBuilder,
        FilterGroupBuilder $filterGroupBuilder,
        array $fieldTranslatorArray = []
    ) {
        $this->filterBuilder = $filterBuilder;
        $this->filterGroupBuilder = $filterGroupBuilder;
        $this->fieldTranslatorArray = array_replace($this->fieldTranslatorArray, $fieldTranslatorArray);
    }

    /**
     * Create filter for filtering the requested categories id's based on url_key, ids, name in the result.
     *
     * @return FilterGroup[]
     */
    public function createFilterGroups(
        array $args,
        int $userId,
        int $storeId
    ): array {
        $filterGroups = [];
        $this->filterGroupBuilder->setFilters([$this->createFilter('customer_id', $userId)]);
        $filterGroups[] = $this->filterGroupBuilder->create();

        $this->filterGroupBuilder->setFilters([$this->createFilter('store_id', $storeId)]);
        $filterGroups[] = $this->filterGroupBuilder->create();

        if (isset($args['filter'])) {
            $filters = [];
            foreach ($args['filter'] as $field => $cond) {
                if (isset($this->fieldTranslatorArray[$field])) {
                    $field = $this->fieldTranslatorArray[$field];
                }
                foreach ($cond as $condType => $value) {
                    if ($condType === 'match') {
                        if (is_array($value)) {
                            throw new InputException(__('Invalid match filter'));
                        }
                        $searchValue = $value !== null ? str_replace('%', '', $value) : '';
                        $filters[] = $this->createFilter($field, "%{$searchValue}%", 'like');
                    } else {
                        $filters[] = $this->createFilter($field, $value, $condType);
                    }
                }
            }

            $this->filterGroupBuilder->setFilters($filters);
            $filterGroups[] = $this->filterGroupBuilder->create();
        }

        return $filterGroups;
    }

    /**
     * @param string $field
     * @param string|int $value
     * @param string $conditionType
     */
    private function createFilter(string $field, $value, string $conditionType = 'eq'): Filter
    {
        $this->filterBuilder->setField($field);
        $this->filterBuilder->setValue($value);
        $this->filterBuilder->setConditionType($conditionType);
        return $this->filterBuilder->create();
    }
}
