<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver;

use Amasty\RequestAQuoteGraphql\Model\Formatter\Customer\Quote as QuoteFormatter;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\IsEnabled as IsQuoteCartEnabledProvider;
use Amasty\RequestAQuoteGraphql\Model\Resolver\CustomerQuotes\Query\QuoteFilter;
use Amasty\RequestQuote\Api\Data\QuoteInterface;
use Amasty\RequestQuote\Api\Data\QuoteSearchResultsInterface;
use Amasty\RequestQuote\Api\QuoteRepositoryInterface;
use Amasty\RequestQuote\Model\Source\Status as StatusConfig;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\InputException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\GraphQl\Model\Query\Context;
use Magento\Store\Api\Data\StoreInterface;

class CustomerQuotes implements ResolverInterface
{
    /**
     * @var QuoteRepositoryInterface
     */
    private $quoteRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var QuoteFilter
     */
    private $quoteFilter;

    /**
     * @var QuoteFormatter
     */
    private $quoteFormatter;

    /**
     * @var StatusConfig
     */
    private $statusConfig;

    /**
     * @var IsQuoteCartEnabledProvider
     */
    private $isQuoteCartEnabled;

    public function __construct(
        IsQuoteCartEnabledProvider $isQuoteCartEnabled,
        QuoteRepositoryInterface $quoteRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        QuoteFilter $quoteFilter,
        QuoteFormatter $quoteFormatter,
        StatusConfig $statusConfig
    ) {
        $this->isQuoteCartEnabled = $isQuoteCartEnabled;
        $this->quoteRepository = $quoteRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->quoteFilter = $quoteFilter;
        $this->quoteFormatter = $quoteFormatter;
        $this->statusConfig = $statusConfig;
    }

    /**
     * @param Field $field
     * @param Context $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlAuthorizationException
     * @throws GraphQlInputException
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$this->isQuoteCartEnabled->execute()) {
            throw new GraphQlAuthorizationException(__('Quote Cart has been disabled.'));
        }

        if (!$context->getExtensionAttributes()->getIsCustomer()) {
            throw new GraphQlAuthorizationException(__('The current customer isn\'t authorized.'));
        }
        if ($args['currentPage'] < 1) {
            throw new GraphQlInputException(__('currentPage value must be greater than 0.'));
        }
        if ($args['pageSize'] < 1) {
            throw new GraphQlInputException(__('pageSize value must be greater than 0.'));
        }

        $userId = $context->getUserId();
        /** @var StoreInterface $store */
        $store = $context->getExtensionAttributes()->getStore();
        try {
            $searchResult = $this->getSearchResult($args, (int)$userId, (int) $store->getId());
            $maxPages = (int)ceil($searchResult->getTotalCount() / $searchResult->getSearchCriteria()->getPageSize());
        } catch (InputException $e) {
            throw new GraphQlInputException(__($e->getMessage()));
        }

        $quotesArray = [];
        foreach ($searchResult->getItems() as $quoteModel) {
            $quotesArray[] = $this->quoteFormatter->format($quoteModel);
        }

        return [
            'total_count' => $searchResult->getTotalCount(),
            'items' => $quotesArray,
            'page_info' => [
                'page_size' => $searchResult->getSearchCriteria()->getPageSize(),
                'current_page' => $searchResult->getSearchCriteria()->getCurrentPage(),
                'total_pages' => $maxPages,
            ]
        ];
    }

    private function getSearchResult(array $args, int $userId, int $storeId): QuoteSearchResultsInterface
    {
        $filterGroups = $this->quoteFilter->createFilterGroups($args, $userId, (int)$storeId);
        $this->searchCriteriaBuilder->setFilterGroups($filterGroups);
        $this->searchCriteriaBuilder->addFilter(
            QuoteInterface::STATUS,
            $this->statusConfig->getVisibleOnFrontStatuses(),
            'in'
        );
        if (isset($args['currentPage'])) {
            $this->searchCriteriaBuilder->setCurrentPage($args['currentPage']);
        }
        if (isset($args['pageSize'])) {
            $this->searchCriteriaBuilder->setPageSize($args['pageSize']);
        }

        return $this->quoteRepository->getRequestsList($this->searchCriteriaBuilder->create());
    }
}
