<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver;

use Amasty\RequestAQuoteGraphql\Model\QuoteCart\CreateEmptyQuoteCartForCustomer;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\CreateEmptyQuoteCartForGuest;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\IsEnabledForCustomer as IsQuoteCartEnabledProvider;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAlreadyExistsException;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\GraphQl\Model\Query\ContextInterface;
use Magento\Quote\Model\MaskedQuoteIdToQuoteIdInterface;

class CreateEmptyQuoteCart implements ResolverInterface
{
    /**
     * @var CreateEmptyQuoteCartForCustomer
     */
    private $createEmptyQuoteCartForCustomer;

    /**
     * @var CreateEmptyQuoteCartForGuest
     */
    private $createEmptyQuoteCartForGuest;

    /**
     * @var MaskedQuoteIdToQuoteIdInterface
     */
    private $maskedQuoteIdToQuoteId;

    /**
     * @var IsQuoteCartEnabledProvider
     */
    private $isQuoteCartEnabled;

    public function __construct(
        CreateEmptyQuoteCartForCustomer $createEmptyCartForCustomer,
        CreateEmptyQuoteCartForGuest $createEmptyCartForGuest,
        MaskedQuoteIdToQuoteIdInterface $maskedQuoteIdToQuoteId,
        IsQuoteCartEnabledProvider $isQuoteCartEnabled
    ) {
        $this->createEmptyQuoteCartForCustomer = $createEmptyCartForCustomer;
        $this->createEmptyQuoteCartForGuest = $createEmptyCartForGuest;
        $this->maskedQuoteIdToQuoteId = $maskedQuoteIdToQuoteId;
        $this->isQuoteCartEnabled = $isQuoteCartEnabled;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlAuthorizationException
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        if (!$this->isQuoteCartEnabled->execute($context->getExtensionAttributes()->getCustomerGroupId())) {
            throw new GraphQlAuthorizationException(__('Quote Cart has been disabled.'));
        }

        $customerId = $context->getUserId();
        $predefinedMaskedQuoteId = null;

        if (isset($args['input']['quote_mask_id'])) {
            $predefinedMaskedQuoteId = $args['input']['quote_mask_id'];
            $this->validateMaskedId($predefinedMaskedQuoteId);
        }

        $maskedQuoteId = $customerId
            ? $this->createEmptyQuoteCartForCustomer->execute($customerId, $predefinedMaskedQuoteId)
            : $this->createEmptyQuoteCartForGuest->execute($predefinedMaskedQuoteId);

        return $maskedQuoteId;
    }

    /**
     * Validate masked id
     *
     * @param string $maskedId
     * @throws GraphQlAlreadyExistsException
     * @throws GraphQlInputException
     */
    private function validateMaskedId(string $maskedId): void
    {
        if (mb_strlen($maskedId) != 32) {
            throw new GraphQlInputException(__('Cart ID length should to be 32 symbols.'));
        }

        if ($this->isQuoteWithSuchMaskedIdAlreadyExists($maskedId)) {
            throw new GraphQlAlreadyExistsException(__('Cart with ID "%1" already exists.', $maskedId));
        }
    }

    /**
     * Check is quote with such maskedId already exists
     */
    private function isQuoteWithSuchMaskedIdAlreadyExists(string $maskedId): bool
    {
        try {
            $this->maskedQuoteIdToQuoteId->execute($maskedId);
            return true;
        } catch (NoSuchEntityException $e) {
            return false;
        }
    }
}
