<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver;

use Amasty\RequestAQuoteGraphql\Model\QuoteCart\IsEnabled as IsQuoteCartEnabledProvider;
use Amasty\RequestQuote\Api\Data\QuoteInterface;
use Amasty\RequestQuote\Api\QuoteRepositoryInterface as AmastyQuoteRepository;
use Amasty\RequestQuote\Model\Email\Sender as EmailSender;
use Amasty\RequestQuote\Model\Source\Status;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\GraphQl\Model\Query\ContextInterface;

class CancelQuote implements ResolverInterface
{
    /**
     * @var IsQuoteCartEnabledProvider
     */
    private $isQuoteCartEnabled;

    /**
     * @var AmastyQuoteRepository
     */
    private $amastyQuoteRepository;

    /**
     * @var EmailSender
     */
    private $emailSender;

    public function __construct(
        IsQuoteCartEnabledProvider $isQuoteCartEnabled,
        AmastyQuoteRepository $amastyQuoteRepository,
        EmailSender $emailSender
    ) {
        $this->isQuoteCartEnabled = $isQuoteCartEnabled;
        $this->amastyQuoteRepository = $amastyQuoteRepository;
        $this->emailSender = $emailSender;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return bool
     * @throws GraphQlAuthorizationException
     * @throws GraphQlInputException
     * @throws GraphQlNoSuchEntityException
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */

    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$this->isQuoteCartEnabled->execute()) {
            throw new GraphQlAuthorizationException(__('Quote Cart has been disabled.'));
        }

        if (false === $context->getExtensionAttributes()->getIsCustomer()) {
            throw new GraphQlAuthorizationException(__('The request is allowed for logged in customer'));
        }

        if (empty($args['quoteUid'])) {
            throw new GraphQlInputException(__('Required parameter "quoteUid" is missing'));
        }

        // phpcs:ignore Magento2.Functions.DiscouragedFunction
        $quoteId = (int) base64_decode($args['quoteUid'], true);
        if (!$quoteId) {
            throw new GraphQlInputException(__('Value of uid "%1" is incorrect.', $args['quoteUid']));
        }

        try {
            /** @var QuoteInterface $quote */
            $quote = $this->amastyQuoteRepository->get($quoteId);
        } catch (NoSuchEntityException $e) {
            throw new GraphQlNoSuchEntityException(
                __('Could not find a quote with UID "%quote_uid"', ['quote_uid' => $args['quoteUid']])
            );
        }

        if ((int) $quote->getCustomerId() !== $context->getUserId()) {
            throw new GraphQlAuthorizationException(
                __(
                    'The current user cannot perform operations on quote "%quote_uid"',
                    ['quote_uid' => $args['quoteUid']]
                )
            );
        }

        if (!in_array($quote->getStatus(), [Status::PENDING, Status::APPROVED], true)) {
            throw new GraphQlAuthorizationException(__('Can\'t close Request Quote'));
        }

        $quote->setStatus(Status::CANCELED);
        $this->amastyQuoteRepository->save($quote);

        $this->emailSender->sendDeclineEmail($quote);

        return true;
    }
}
