<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver;

use Amasty\Base\Model\Serializer;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\IsEnabledForCustomer as IsQuoteCartEnabledProvider;
use Amasty\RequestQuote\Api\Data\QuoteItemInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\GraphQl\Model\Query\Context;
use Magento\Quote\Api\Data\CartItemInterface;
use Magento\Quote\Model\Quote\Item as QuoteItem;

class AmastyQuoteItemFields implements ResolverInterface
{
    /**
     * @var IsQuoteCartEnabledProvider
     */
    private $isQuoteCartEnabled;

    /**
     * @var Serializer
     */
    private $serializer;

    public function __construct(
        IsQuoteCartEnabledProvider $isQuoteCartEnabled,
        Serializer $serializer
    ) {
        $this->isQuoteCartEnabled = $isQuoteCartEnabled;
        $this->serializer = $serializer;
    }

    /**
     * @param Field $field
     * @param Context $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws LocalizedException
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$this->isQuoteCartEnabled->execute($context->getExtensionAttributes()->getCustomerGroupId())) {
            throw new GraphQlAuthorizationException(__('Quote Cart has been disabled.'));
        }

        if (!isset($value['model'])) {
            throw new LocalizedException(__('"model" value should be specified'));
        }

        /** @var CartItemInterface|QuoteItem $quoteItem */
        $quoteItem = $value['model'];

        $quoteItemAdditionalData = $this->serializer->unserialize($quoteItem->getAdditionalData());

        return [
            'model' => $quoteItem,
            'notes' => [
                'customer_note' => $quoteItemAdditionalData[QuoteItemInterface::CUSTOMER_NOTE_KEY] ?? '',
                'admin_note' => $quoteItemAdditionalData[QuoteItemInterface::ADMIN_NOTE_KEY] ?? ''
            ]
        ];
    }
}
