<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver;

use Amasty\Base\Model\Serializer;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\IsEnabledForCustomer as IsQuoteCartEnabledProvider;
use Amasty\RequestQuote\Api\Data\QuoteInterface;
use Amasty\RequestQuote\Model\Quote\Frontend\GetAmastyQuote;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\GraphQl\Model\Query\Context;

class AmastyQuoteFields implements ResolverInterface
{
    /**
     * @var IsQuoteCartEnabledProvider
     */
    private $isQuoteCartEnabled;

    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var GetAmastyQuote
     */
    private $getAmastyQuote;

    public function __construct(
        IsQuoteCartEnabledProvider $isQuoteCartEnabled,
        Serializer $serializer,
        GetAmastyQuote $getAmastyQuote
    ) {
        $this->serializer = $serializer;
        $this->isQuoteCartEnabled = $isQuoteCartEnabled;
        $this->getAmastyQuote = $getAmastyQuote;
    }

    /**
     * @param Field $field
     * @param Context $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws LocalizedException
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        if (!$this->isQuoteCartEnabled->execute($context->getExtensionAttributes()->getCustomerGroupId())) {
            throw new GraphQlAuthorizationException(__('Quote Cart has been disabled.'));
        }

        if (!isset($value['model'])) {
            throw new LocalizedException(__('"model" value should be specified'));
        }

        $amastyQuote = $this->getAmastyQuote->execute($value['model']) ?? $value['model'];

        if ($amastyQuote->getRemarks()) {
            $remarks = $this->serializer->unserialize($amastyQuote->getRemarks());
        } else {
            $remarks = [];
        }

        return [
            'status' => $amastyQuote->hasData(QuoteInterface::STATUS)
                ? (int) $amastyQuote->getData(QuoteInterface::STATUS)
                : null,
            'notes' => [
                'customer_note' => $remarks[QuoteInterface::CUSTOMER_NOTE_KEY] ?? '',
                'admin_note' => $remarks[QuoteInterface::ADMIN_NOTE_KEY] ?? ''
            ],
            'shipping_can_modified' => $amastyQuote->hasData(QuoteInterface::SHIPPING_CAN_BE_MODIFIED)
                ? $amastyQuote->getData(QuoteInterface::SHIPPING_CAN_BE_MODIFIED)
                : null,
            'shipping_configured' => $amastyQuote->hasData(QuoteInterface::SHIPPING_CONFIGURE)
                ? $amastyQuote->getData(QuoteInterface::SHIPPING_CONFIGURE)
                : null
        ];
    }
}
