<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Resolver;

use Amasty\RequestAQuoteGraphql\Model\Di\Wrapper as DiWrapper;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\GetQuoteForUser;
use Amasty\RequestAQuoteGraphql\Model\QuoteCart\IsEnabledForCustomer as IsQuoteCartEnabledProvider;
use Amasty\RequestQuote\Model\QuoteMutexInterface;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\GraphQl\Model\Query\ContextInterface;
use Magento\Quote\Model\Cart\AddProductsToCart as AddProductsToQuoteCart24;
use Magento\Quote\Model\Cart\Data\CartItemFactory;
use Magento\Quote\Model\Cart\Data\Error;
use Magento\QuoteGraphQl\Model\Cart\AddProductsToCart as AddProductsToQuoteCart23;
use Magento\QuoteGraphQl\Model\CartItem\DataProvider\Processor\ItemDataProcessorInterface;

class AddProductsToQuoteCart implements ResolverInterface
{
    /**
     * @var GetQuoteForUser
     */
    private $getQuoteForUser;

    /**
     * @var ItemDataProcessorInterface|DiWrapper
     */
    private $itemDataProcessor;

    /**
     * @var QuoteMutexInterface
     */
    private $quoteMutex;

    /**
     * @var DiWrapper|CartItemFactory
     */
    private $cartItemFactory;

    /**
     * @var DiWrapper|AddProductsToQuoteCart23
     */
    private $addProductsToQuoteCart23;

    /**
     * @var DiWrapper|AddProductsToQuoteCart24
     */
    private $addProductsToQuoteCart24;

    /**
     * @var IsQuoteCartEnabledProvider
     */
    private $isQuoteCartEnabled;

    public function __construct(
        GetQuoteForUser $getQuoteForUser,
        DiWrapper $itemDataProcessor,
        QuoteMutexInterface $quoteMutex,
        DiWrapper $cartItemFactory,
        DiWrapper $addProductsToQuoteCart23,
        DiWrapper $addProductsToQuoteCart24,
        IsQuoteCartEnabledProvider $isQuoteCartEnabled
    ) {
        $this->getQuoteForUser = $getQuoteForUser;
        $this->itemDataProcessor = $itemDataProcessor;
        $this->quoteMutex = $quoteMutex;
        $this->cartItemFactory = $cartItemFactory;
        $this->addProductsToQuoteCart23 = $addProductsToQuoteCart23;
        $this->addProductsToQuoteCart24 = $addProductsToQuoteCart24;
        $this->isQuoteCartEnabled = $isQuoteCartEnabled;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlAuthorizationException
     * @throws GraphQlInputException
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$this->isQuoteCartEnabled->execute($context->getExtensionAttributes()->getCustomerGroupId())) {
            throw new GraphQlAuthorizationException(__('Quote Cart has been disabled.'));
        }

        if (empty($args['quoteId'])) {
            throw new GraphQlInputException(__('Required parameter "quoteId" is missing'));
        }
        if (empty($args['quoteItems']) || !is_array($args['quoteItems'])) {
            throw new GraphQlInputException(__('Required parameter "quoteItems" is missing'));
        }

        return $this->quoteMutex->execute(
            [$args['quoteId']],
            \Closure::fromCallable([$this, 'run']),
            [$context, $args]
        );
    }

    private function run(ContextInterface $context, ?array $args): array
    {
        $maskedQuoteId = $args['quoteId'];
        $quoteItemsData = $args['quoteItems'];
        $storeId = (int)$context->getExtensionAttributes()->getStore()->getId();

        // Shopping Cart validation
        $quoteCart = $this->getQuoteForUser->execute($maskedQuoteId, $context->getUserId(), $storeId);

        if (class_exists(AddProductsToQuoteCart24::class)) {
            $quoteItems = [];
            foreach ($quoteItemsData as $quoteItemData) {
                if (!$this->itemIsAllowedToCart($quoteItemData, $context)) {
                    continue;
                }
                $quoteItems[] = $this->cartItemFactory->create($quoteItemData);
            }
            $addProductsToQuoteOutput = $this->addProductsToQuoteCart24->execute($maskedQuoteId, $quoteItems);

            return [
                'cart' => [
                    'model' => $addProductsToQuoteOutput->getCart(),
                ],
                'user_errors' => array_map(
                    function (Error $error) {
                        return [
                            'code' => $error->getCode(),
                            'message' => $error->getMessage(),
                            'path' => [$error->getCartItemPosition()]
                        ];
                    },
                    $addProductsToQuoteOutput->getErrors()
                )
            ];
        } else {
            $this->addProductsToQuoteCart23->execute($quoteCart, $this->prepareQuoteItemsFor23($quoteItemsData));

            return [
                'cart' => [
                    'model' => $quoteCart,
                ],
                'user_errors' => []
            ];
        }
    }

    /**
     * Check if the item can be added to quote
     */
    private function itemIsAllowedToCart(array $quoteItemData, ContextInterface $context): bool
    {
        $quoteItemData = $this->itemDataProcessor->process($quoteItemData, $context);

        if (isset($quoteItemData['grant_checkout']) && $quoteItemData['grant_checkout'] === false) {
            return false;
        }

        return true;
    }

    private function prepareQuoteItemsFor23(array $quoteItemsData): array
    {
        foreach ($quoteItemsData as &$quoteItemData) {
            $quoteItemData['data'] = [];
            if (isset($quoteItemData['sku'])) {
                $quoteItemData['data']['sku'] = $quoteItemData['sku'];
            }
            if (isset($quoteItemData['quantity'])) {
                $quoteItemData['data']['quantity'] = $quoteItemData['quantity'];
            }
        }

        return $quoteItemsData;
    }
}
