<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\QuoteItem\DataProvider;

use Amasty\RequestAQuoteGraphql\Model\Di\Wrapper as DiWrapper;
use Amasty\RequestQuote\Model\Quote;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Quote\Api\CartItemRepositoryInterface;
use Magento\QuoteGraphQl\Model\Cart\UpdateCartItem;
use Magento\QuoteGraphQl\Model\CartItem\DataProvider\UpdateCartItems as UpdateCartItemsProvider;

/**
 * Used for m23.
 */
class UpdateQuoteItems
{
    /**
     * @var CartItemRepositoryInterface
     */
    private $cartItemRepository;

    /**
     * @var UpdateCartItem
     */
    private $updateCartItem;

    /**
     * @var UpdateCartItemsProvider|DiWrapper
     */
    private $updateCartItemsProvider;

    public function __construct(
        CartItemRepositoryInterface $cartItemRepository,
        UpdateCartItem $updateCartItem,
        DiWrapper $updateCartItemsProvider
    ) {
        $this->cartItemRepository = $cartItemRepository;
        $this->updateCartItem = $updateCartItem;
        $this->updateCartItemsProvider = $updateCartItemsProvider;
    }

    public function execute(Quote $quote, array $quoteItemsData): void
    {
        // Try call magento cart item updater. If class does not exist, run custom updater.
        if ($this->updateCartItemsProvider->processCartItems($quote, $quoteItemsData) === false) {
            $this->processCartItems($quote, $quoteItemsData);
        }
    }

    /**
     * @throws GraphQlInputException
     */
    private function processCartItems(Quote $quote, array $quoteItemsData): void
    {
        foreach ($quoteItemsData as $quoteItemData) {
            if (empty($quoteItemData['cart_item_id'])) {
                throw new GraphQlInputException(__('Required parameter "cart_item_id" for "cart_items" is missing.'));
            }
            $itemId = (int) $quoteItemData['cart_item_id'];
            $customizableOptions = $quoteItemData['customizable_options'] ?? [];

            $quoteItem = $quote->getItemById($itemId);
            if ($quoteItem && $quoteItem->getParentItemId()) {
                throw new GraphQlInputException(__('Child items may not be updated.'));
            }

            if (count($customizableOptions) === 0 && !isset($quoteItemData['quantity'])) {
                throw new GraphQlInputException(__('Required parameter "quantity" for "cart_items" is missing.'));
            }

            $quantity = (float) $quoteItemData['quantity'];
            if ($quantity <= 0.0) {
                $this->cartItemRepository->deleteById((int) $quote->getId(), $itemId);
            } else {
                $this->updateCartItem->execute($quote, $itemId, $quantity, $customizableOptions);
            }
        }
    }
}
