<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\QuoteCart\SubmitQuote;

use Amasty\RequestAQuoteGraphql\Model\Di\Wrapper as DiWrapper;
use Magento\CustomerGraphQl\Model\Customer\CreateCustomerAccount;
use Magento\Framework\Event\ManagerInterface;
use Magento\Integration\Api\UserTokenIssuerInterface;
use Magento\Integration\Model\CustomUserContext;
use Magento\Integration\Model\CustomUserContextFactory;
use Magento\Integration\Model\Oauth\TokenFactory;
use Magento\Integration\Model\UserToken\UserTokenParametersFactory;
use Magento\Store\Api\Data\StoreInterface;

class GenerateCustomerToken
{
    /**
     * @var CreateCustomerAccount
     */
    private $createCustomerAccount;

    /**
     * @var ManagerInterface
     */
    private $eventManager;

    /**
     * @var DiWrapper|CustomUserContextFactory
     */
    private $customerUserContextFactory;

    /**
     * @var DiWrapper|UserTokenParametersFactory
     */
    private $userTokenParametersFactory;

    /**
     * @var DiWrapper|UserTokenIssuerInterface
     */
    private $userTokenIssuer;

    /**
     * @var DiWrapper|TokenFactory
     */
    private $tokenFactory;

    /**
     * @var CustomerTokenFactory
     */
    private $customerTokenFactory;

    public function __construct(
        CreateCustomerAccount $createCustomerAccount,
        ManagerInterface $eventManager,
        DiWrapper $customerUserContextFactory,
        DiWrapper $userTokenParametersFactory,
        DiWrapper $userTokenIssuer,
        DiWrapper $tokenFactory,
        CustomerTokenFactory $customerTokenFactory
    ) {
        $this->createCustomerAccount = $createCustomerAccount;
        $this->eventManager = $eventManager;
        $this->customerUserContextFactory = $customerUserContextFactory;
        $this->userTokenParametersFactory = $userTokenParametersFactory;
        $this->userTokenIssuer = $userTokenIssuer;
        $this->tokenFactory = $tokenFactory;
        $this->customerTokenFactory = $customerTokenFactory;
    }

    public function execute(array $customerData, StoreInterface $store): CustomerToken
    {
        if (isset($customerData['date_of_birth'])) {
            $customerData['dob'] = $customerData['date_of_birth'];
        }

        $customer = $this->createCustomerAccount->execute($customerData, $store);

        $this->eventManager->dispatch('customer_login', ['customer' => $customer]);

        $context = $this->customerUserContextFactory->create([
            'userId' => (int) $customer->getId(),
            'userType' => CustomUserContext::USER_TYPE_CUSTOMER
        ]);
        if ($context) {
            $params = $this->userTokenParametersFactory->create();
            $token = $this->userTokenIssuer->create($context, $params);
        } else {
            $tokenModel = $this->tokenFactory->create();
            $token = $tokenModel->createCustomerToken($customer->getId())->getToken();
        }

        return $this->customerTokenFactory->create([
            'customer' => $customer,
            'token' => $token
        ]);
    }
}
