<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\QuoteCart;

use Amasty\RequestQuote\Api\QuoteRepositoryInterface;
use Amasty\RequestQuote\Model\Quote;
use Amasty\RequestQuote\Model\Source\Status;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Quote\Model\MaskedQuoteIdToQuoteIdInterface;
use Magento\Store\Api\StoreRepositoryInterface;

class GetQuoteForUser
{
    /**
     * @var MaskedQuoteIdToQuoteIdInterface
     */
    private $maskedQuoteIdToQuoteId;

    /**
     * @var QuoteRepositoryInterface
     */
    private $quoteRepository;

    /**
     * @var StoreRepositoryInterface
     */
    private $storeRepository;

    public function __construct(
        MaskedQuoteIdToQuoteIdInterface $maskedQuoteIdToQuoteId,
        QuoteRepositoryInterface $quoteRepository,
        StoreRepositoryInterface $storeRepository = null
    ) {
        $this->maskedQuoteIdToQuoteId = $maskedQuoteIdToQuoteId;
        $this->quoteRepository = $quoteRepository;
        $this->storeRepository = $storeRepository ?: ObjectManager::getInstance()->get(StoreRepositoryInterface::class);
    }

    /**
     * @throws GraphQlAuthorizationException
     * @throws GraphQlInputException
     * @throws GraphQlNoSuchEntityException
     */
    public function execute(string $cartHash, ?int $customerId, int $storeId): Quote
    {
        try {
            $quoteId = $this->maskedQuoteIdToQuoteId->execute($cartHash);
        } catch (NoSuchEntityException $exception) {
            throw new GraphQlNoSuchEntityException(
                __('Could not find a cart with ID "%masked_cart_id"', ['masked_cart_id' => $cartHash])
            );
        }

        try {
            /** @var Quote $quote */
            $quote = $this->quoteRepository->get($quoteId);
        } catch (NoSuchEntityException $e) {
            throw new GraphQlNoSuchEntityException(
                __('Could not find a cart with ID "%masked_cart_id"', ['masked_cart_id' => $cartHash])
            );
        }

        if ($quote->getStatus() !== Status::CREATED) {
            throw new GraphQlNoSuchEntityException(__('The quote cart can\'t be modified'));
        }

        $quote = $this->updateCartCurrency($quote, $storeId);
        $quoteCustomerId = (int)$quote->getCustomerId();

        /* Guest cart, allow operations */
        if (0 === $quoteCustomerId && (null === $customerId || 0 === $customerId)) {
            return $quote;
        }

        if ($quoteCustomerId !== $customerId) {
            throw new GraphQlAuthorizationException(
                __(
                    'The current user cannot perform operations on cart "%masked_cart_id"',
                    ['masked_cart_id' => $cartHash]
                )
            );
        }

        return $quote;
    }

    /**
     * Sets cart currency based on specified store.
     */
    private function updateCartCurrency(Quote $quote, int $storeId): Quote
    {
        $cartStore = $this->storeRepository->getById($quote->getStoreId());
        $currentCartCurrencyCode = $cartStore->getCurrentCurrency()->getCode();

        if ((int)$quote->getStoreId() !== $storeId) {
            $newStore = $this->storeRepository->getById($storeId);
            if ($cartStore->getWebsite() !== $newStore->getWebsite()) {
                throw new GraphQlInputException(
                    __('Can\'t assign cart to store in different website.')
                );
            }
            $quote->setStoreId($storeId);
            $quote->setStoreCurrencyCode($newStore->getCurrentCurrency()->getCode());
            $quote->setQuoteCurrencyCode($newStore->getCurrentCurrency()->getCode());
        } elseif ($quote->getQuoteCurrencyCode() !== $currentCartCurrencyCode) {
            $quote->setQuoteCurrencyCode($cartStore->getCurrentCurrency()->getCode());
        } else {
            return $quote;
        }

        $this->quoteRepository->save($quote);
        $quote = $this->quoteRepository->get($quote->getId());

        return $quote;
    }
}
