<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\QuoteCart;

use Amasty\RequestAQuoteGraphql\Model\QuoteManagement;
use Amasty\RequestQuote\Model\Quote;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Model\QuoteIdMaskFactory;
use Magento\Quote\Model\QuoteIdToMaskedQuoteIdInterface;
use Magento\Quote\Model\ResourceModel\Quote\QuoteIdMask as QuoteIdMaskResourceModel;

class CustomerQuoteResolver
{
    /**
     * @var QuoteIdToMaskedQuoteIdInterface
     */
    private $quoteIdToMaskedQuoteId;

    /**
     * @var QuoteIdMaskFactory
     */
    private $quoteIdMaskFactory;

    /**
     * @var QuoteIdMaskResourceModel
     */
    private $quoteIdMaskResourceModel;

    /**
     * @var QuoteManagement
     */
    private $quoteManagement;

    public function __construct(
        QuoteIdToMaskedQuoteIdInterface $quoteIdToMaskedQuoteId,
        QuoteIdMaskFactory $quoteIdMaskFactory,
        QuoteIdMaskResourceModel $quoteIdMaskResourceModel,
        QuoteManagement $quoteManagement
    ) {
        $this->quoteIdToMaskedQuoteId = $quoteIdToMaskedQuoteId;
        $this->quoteIdMaskFactory = $quoteIdMaskFactory;
        $this->quoteIdMaskResourceModel = $quoteIdMaskResourceModel;
        $this->quoteManagement = $quoteManagement;
    }

    public function resolve(int $customerId, string $predefinedMaskedQuoteId = null): Quote
    {
        try {
            /** @var Quote $quote */
            $quote = $this->quoteManagement->getQuoteForCustomer($customerId);
        } catch (NoSuchEntityException $e) {
            $this->quoteManagement->createEmptyQuoteForCustomer($customerId);
            $quote = $this->quoteManagement->getQuoteForCustomer($customerId);
        }

        try {
            $this->ensureQuoteMaskIdExist((int)$quote->getId(), $predefinedMaskedQuoteId);
            // phpcs:ignore Magento2.CodeAnalysis.EmptyBlock
        } catch (AlreadyExistsException $e) {
            // do nothing, we already have masked id
        }

        return $quote;
    }

    /**
     * Create masked id for customer's active quote if it's not exists
     */
    private function ensureQuoteMaskIdExist(int $quoteId, string $predefinedMaskedQuoteId = null): void
    {
        try {
            $maskedId = $this->quoteIdToMaskedQuoteId->execute($quoteId);
        } catch (NoSuchEntityException $e) {
            $maskedId = '';
        }
        if ($maskedId === '') {
            $quoteIdMask = $this->quoteIdMaskFactory->create();
            $quoteIdMask->setQuoteId($quoteId);
            if (null !== $predefinedMaskedQuoteId) {
                $quoteIdMask->setMaskedId($predefinedMaskedQuoteId);
            }
            $this->quoteIdMaskResourceModel->save($quoteIdMask);
        }
    }
}
