<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Model\Formatter\Customer;

use Amasty\Base\Model\Serializer;
use Amasty\RequestQuote\Api\Data\QuoteItemInterface;
use Amasty\RequestQuote\Model\Source\Status;
use Magento\Framework\Pricing\Adjustment\Calculator as PriceCalculator;
use Magento\Framework\Pricing\PriceCurrencyInterface;
use Magento\Quote\Api\Data\CartItemInterface;

class QuoteItem
{
    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var PriceCalculator
     */
    private $priceCalculator;

    /**
     * @var PriceCurrencyInterface
     */
    private $priceCurrency;

    public function __construct(
        Serializer $serializer,
        PriceCalculator $priceCalculator,
        PriceCurrencyInterface $priceCurrency
    ) {
        $this->serializer = $serializer;
        $this->priceCalculator = $priceCalculator;
        $this->priceCurrency = $priceCurrency;
    }

    public function format(CartItemInterface $quoteItem): array
    {
        $quoteItemAdditionalData = $this->serializer->unserialize($quoteItem->getAdditionalData());
        $currency = $quoteItem->getQuote()->getQuoteCurrencyCode();
        $requestedPrice = (float) ($quoteItemAdditionalData[QuoteItemInterface::REQUESTED_PRICE] ?? 0);
        $isApprovedPriceShow = in_array(
            $quoteItem->getQuote()->getStatus(),
            [Status::APPROVED, Status::EXPIRED, Status::COMPLETE],
            true
        );
        [$discountAmount, $discountPercent] = $this->getDiscounts($quoteItem);
        $isHidePrice = $quoteItemAdditionalData[QuoteItemInterface::HIDE_ORIGINAL_PRICE] ?? false;

        return [
            'id' => base64_encode($quoteItem->getId()),
            'product_name' => $quoteItem->getName(),
            'sku' => $quoteItem->getSku(),
            'quantity' => (float) $quoteItem->getQty(),
            'notes' => [
                'customer_note' => $quoteItemAdditionalData[QuoteItemInterface::CUSTOMER_NOTE_KEY] ?? '',
                'admin_note' => $quoteItemAdditionalData[QuoteItemInterface::ADMIN_NOTE_KEY] ?? ''
            ],
            'hide_price' => $isHidePrice,
            'quote_discount_amount' => [
                'currency' => $currency,
                'value' => $isHidePrice ? null : $this->convertPrice((float) $discountAmount, $quoteItem)
            ],
            'quote_discount_percent' => $isHidePrice ? null : sprintf("%.2f", $discountPercent),
            'original_price' => [
                'currency' => $currency,
                'value' => $isHidePrice ? null : $this->convertPrice($this->getOriginalPrice($quoteItem), $quoteItem)
            ],
            'requested_price' => [
                'currency' => $currency,
                'value' => $this->convertPrice((float) $requestedPrice, $quoteItem)
            ],
            'approved_price' => [
                'currency' => $currency,
                'value' => $isApprovedPriceShow ? $this->convertPrice((float) $quoteItem->getPrice(), $quoteItem) : 0
            ],
            'subtotal' => [
                'currency' => $currency,
                'value' => (float) $quoteItem->getRowTotal()
            ]
        ];
    }

    private function getDiscounts(CartItemInterface $quoteItem): array
    {
        $approvedPrice = $quoteItem->getPrice();
        $originalPrice = $this->getOriginalPrice($quoteItem);
        $discountAmount = ($originalPrice - $approvedPrice) * $quoteItem->getQty();
        if ($originalPrice) {
            $discountPercent = ($originalPrice - $approvedPrice) / $originalPrice * 100;
        } else {
            $discountPercent = 0;
        }

        return [$discountAmount, $discountPercent];
    }

    private function getOriginalPrice(CartItemInterface $quoteItem): float
    {
        return $this->priceCalculator->getAmount(
            $quoteItem->getProduct()->getFinalPrice(),
            $quoteItem->getProduct()
        )->getValue(['tax', 'weee_tax']);
    }

    private function convertPrice(float $price, CartItemInterface $quoteItem): float
    {
        $currencyObject = $quoteItem->getQuote()->getCurrency();
        $quoteStore = $quoteItem->getQuote()->getStore();

        return $this->priceCurrency
            ->getCurrency($quoteStore, $currencyObject->getBaseCurrencyCode())
            ->convert($price, $currencyObject->getQuoteCurrencyCode());
    }
}
