<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Advanced Reports Convert To Report Builder for Magento 2
*/

declare(strict_types=1);

namespace Amasty\ReportsToBuilder\Model;

use Amasty\ReportBuilder\Api\ColumnInterface;
use Amasty\ReportBuilder\Api\Data\AxisInterface;
use Amasty\ReportBuilder\Api\Data\ChartInterface;
use Amasty\ReportBuilder\Api\Data\ReportColumnInterface;
use Amasty\ReportBuilder\Api\Data\ReportInterface;
use Amasty\ReportBuilder\Model\Chart\GetAvailableChartTypes;
use Amasty\ReportBuilder\Model\ReportFactory;
use Amasty\ReportBuilder\Ui\DataProvider\Form\Report\Modifier\AddChartData;
use Amasty\ReportsToBuilder\Model\ReportResolver\RegisterColumn;
use Amasty\ReportsToBuilder\Model\ReportResolver\SystemConfigColumn;

class ReportResolver
{
    public const REQUIRED_COLUMN_KEYS = [
        ColumnInterface::ORDER => 0,
        ColumnInterface::VISIBILITY => true,
        ColumnInterface::IS_DATE_FILTER => false,
        ColumnInterface::CUSTOM_TITLE => '',
        ColumnInterface::FILTER => '',
    ];

    /**
     * @var ReportFactory
     */
    private $reportFactory;

    /**
     * @var ReportDataRepository
     */
    private $dataRepository;

    /**
     * @var SystemConfigColumn
     */
    private $configColumn;

    /**
     * @var RegisterColumn
     */
    private $registerColumn;

    public function __construct(
        ReportFactory $reportFactory,
        ReportDataRepository $dataRepository,
        SystemConfigColumn $configColumn,
        RegisterColumn $registerColumn
    ) {
        $this->reportFactory = $reportFactory;
        $this->dataRepository = $dataRepository;
        $this->configColumn = $configColumn;
        $this->registerColumn = $registerColumn;
    }

    public function getReportByHandle(string $handle): ReportInterface
    {
        $data = $this->dataRepository->get($handle);
        $data[ReportInterface::DISPLAY_CHART] = $data[ReportInterface::DISPLAY_CHART] ?? false;
        $data[ReportInterface::COLUMNS] = $this->fulfillColumnsData($data[ReportInterface::COLUMNS]);
        if (!empty($data[AddChartData::CHART_SCOPE])) {
            $data[ReportInterface::DISPLAY_CHART] = true;
            $data[AddChartData::CHART_SCOPE] = $this->filterChartData($data[AddChartData::CHART_SCOPE]);
        }
        $report = $this->reportFactory->create(['data' => $data]);

        $report->isObjectNew(false);

        return $report;
    }

    private function fulfillColumnsData(array $columns): array
    {
        $position = 0;
        foreach ($columns as $columnId => &$column) {
            if (!empty($column[SystemConfigColumn::PATH_KEY])) {
                $this->configColumn->collectColumnId($column);
            }
            if (!\array_key_exists(ReportColumnInterface::COLUMN_ID, $column)) {
                $column[ReportColumnInterface::COLUMN_ID] = $columnId;
            }
            if (!\array_key_exists(ColumnInterface::POSITION, $column)) {
                $column[ColumnInterface::POSITION] = $position;
            }

            foreach (static::REQUIRED_COLUMN_KEYS as $key => $defaultValue) {
                if (!\array_key_exists($key, $column)) {
                    $column[$key] = $defaultValue;
                }
            }

            $this->registerColumn->register($column);

            $position++;
        }

        return $columns;
    }

    private function filterChartData(array $chart): array
    {
        $chartData = [
            ChartInterface::CHART_TYPE => $chart[ChartInterface::CHART_TYPE] ?? GetAvailableChartTypes::LINEAR
        ];
        foreach ($chart[AddChartData::AXISES_SCOPE] as $axis) {
            $chartData[AddChartData::AXISES_SCOPE][] = [
                AxisInterface::TYPE => $axis[AxisInterface::TYPE],
                AxisInterface::VALUE => $axis[AxisInterface::VALUE]
            ];
        }

        return $chartData;
    }
}
