/**
 * Virtual column context Element data model factory
 */
define([
    'underscore',
    'ko',
    'amrepbuilder_helpers'
], function (_, ko, helpers) {
    'use strict';

    /**
     * @typedef {Object} VirtualContextElement
     * @property {String} category Element category type
     * @property {Number|String} id
     * @property {Number} type data type
     * @property {String[]|null} modifier array of unary operators identifications
     * @property {String|null} operation binary operator identification
     * @property {Number} order
     */
    /**
     * @typedef {VirtualContextElement} ObservableVirtualContextElement
     * @property {Function} type observable
     * @property {Function} operation observable
     */

    return {
        columnTypes: {
            VIRTUAL: 'virtual'
        },

        operandTypes: {
            INPUT: 'input',
            BRACKETS: 'brackets',
            COLUMN: 'column',
            NO_OPERATION: null
        },

        inputTypes: {
            STRING: 'varchar',
            DECIMAL: 'decimal'
        },

        /**
         * Create Element object with observables
         *
         * @param {String} category
         * @param {VirtualContextElement|undefined|null} parent
         * @param {VirtualContextElement|undefined} data
         * @returns {ObservableVirtualContextElement}
         */
        create: function (category, parent, data) {
            var elementData = data || {},
                element = {};

            switch (category) {
                case this.operandTypes.INPUT:
                    element = this._createInput(elementData);
                    break;
                case this.operandTypes.BRACKETS:
                    element = this._createBrackets(elementData);
                    break;
                case this.operandTypes.COLUMN:
                    element = this._createColumn(elementData);
                    break;
                default:
            }

            if (parent && parent.id) {
                element.parentItemId = parent.id;
            }

            return element;
        },

        /**
         * @param {VirtualContextElement[]} elementItems
         * @param {VirtualContextElement|undefined} [parent]
         * @returns {ObservableVirtualContextElement[]}
         */
        adaptContext: function (elementItems, parent) {
            return _.map(elementItems, function (data) {
                return this.create(data.category, parent, data);
            }, this);
        },

        /**
         * @param {VirtualContextElement} data
         * @returns {ObservableVirtualContextElement}
         * @private
         */
        _createElement: function (data) {
            return {
                id: data.id || helpers.getRandomString(),
                type: ko.observable(data.type || ''),
                modifier: ko.observableArray(data.modifier || []),
                operation: ko.observable(data.operation || this.operandTypes.NO_OPERATION)
            };
        },

        /**
         * @param {VirtualContextElement} data
         * @returns {ObservableVirtualContextElement}
         * @private
         */
        _createInput: function (data) {
            return _.extend(this._createElement(data), {
                category: this.operandTypes.INPUT,
                value: ko.observable(data.value || ''),
                htmlType: ko.observable(data.htmlType || '')
            });
        },

        /**
         * @param {VirtualContextElement} data
         * @returns {ObservableVirtualContextElement}
         * @private
         */
        _createColumn: function (data) {
            return _.extend(this._createElement(data), {
                category: this.operandTypes.COLUMN,
                columnData: ko.observable(_.isEmpty(data) ? false : data)
            });
        },

        /**
         * @param {VirtualContextElement} data
         * @returns {ObservableVirtualContextElement}
         * @private
         */
        _createBrackets: function (data) {
            var children = [],
                element = this._createElement(data);

            element.category = this.operandTypes.BRACKETS;
            element.isShowToolbar = ko.observable(_.isEmpty(data));

            if (data.elements) {
                children = this.adaptContext(data.elements, data);
            }

            element.elements = ko.observableArray(children);

            return element;
        }
    };
});
