/**
 * Column toolbar component mixin
 */

define([
    'underscore',
    'uiComponent',
    'ko',
    'amrbvirtual_element_factory',
    'constructor_validator',
    'amrepbuilder_helpers'
], function (_, Component, ko, elementFactory, Validator, helpers) {
    'use strict';

    var mixin = {
        defaults: {
            templates: {
                rectangleIcon: 'Amasty_ReportBuilder/components/icons/rectangle'
            },
            imports: {
                dndZone: 'index = chosen_options:dndZone',
                virtualColumnData: 'index = component_virtual_column:virtualColumn',
                categories: 'index = operators_popup:categories',
                mainEntityValue: 'index = main_entity:currentValue'
            },
            extendComponents: [
                'index = operators_popup',
                'index = entities_list'
            ],
            toolbarButtonPathVirtual: 'Amasty_ReportBuilderVirtual/builder/column/toolbar/buttons/'
        },
        constants: {
            columnTypes: elementFactory.columnTypes
        },

        /**
         * Init observable variables
         *
         * @returns {Object}
         */
        initObservable: function () {
            this._super()
                .observe(['dndZone', 'categories', 'mainEntityValue']);

            this.mainEntityValue.subscribe(function (value) {
                if (value) {
                    this._enableChoosingColumns();
                }
            }.bind(this));

            return this;
        },

        /**
         * Initialize Validator
         *
         * @param {Object} constants
         * @returns {void}
         */
        initValidator: function (constants) {
            this.validator = Validator.init(constants);
        },

        /**
         * Toggle virtual column footer visibility
         *
         * @param {Object} item
         * @returns {void}
         */
        toggleFooter: function (item) {
            var state = !this.chosenOptions.isOverlay();

            this.chosenOptions.isOverlay(state);
            this.toggleColumnsAvailability(state);
            item.isFooterActive(!item.isFooterActive());
            item.isOpen(!item.isOpen());
            this.toggleSorting(item);
            this.operatorsPopup.closePopup();
        },

        /**
         * @returns {void}
         */
        _enableChoosingColumns: function () {
            helpers.resolveComponent('chosenOptions', this).isOverlay(false);
            this.toggleColumnsAvailability(false);
        },

        /**
         * Toggle chosen options drop zone columns sorting possibility
         *
         * @param {Object} item
         * @returns {void}
         */
        toggleSorting: function (item) {
            this.dndZone().sortable(item.isFooterActive() ? 'disable' : 'enable');
        },

        /**
         * Aggregation filter initialization
         *
         * @param {Object} column
         * @returns {void}
         */
        initAggregation: function (column) {
            if (this.chosenOptions.isTypeVirtual(column)) {
                this._disableAggregation(column);

                return;
            }

            this._super();
        },

        /**
         * Init toggle constructor button
         *
         * @param {Object} column
         * @returns {void}
         */
        initConstructor: function (column) {
            var contextElements = [];

            if (column.context) {
                contextElements = elementFactory.adaptContext(column.context);
            }

            this._addVirtualColumnTitle(column);
            column.isTypeVirtual = ko.observable(this.chosenOptions.isTypeVirtual(column));
            column.isOpen = ko.observable(false);
            column.context = ko.observableArray(contextElements);
            column.dataType = ko.observable(column.dataType || 'string');
            column.isShowToolbar = ko.observable(
                !(column.type === this.constants.columnTypes.VIRTUAL && column.context().length)
            );
        },

        /**
         * Add a title for a virtual column
         *
         * @param {Object} column
         * @returns {void}
         */
        _addVirtualColumnTitle: function (column) {
            if (_.isUndefined(column.type) || column.type === this.virtualColumnData.type) {
                column.title = ko.observable(this.virtualColumnData.title);
            }
        },

        /**
         * Init toolbar actions templates
         * @returns {void}
         */
        _initButtonsTemplates: function () {
            this._super();

            this.elems().unshift(this.toolbarButtonPathVirtual + 'toggle');
        },

        /**
         * Disable aggregation button for the virtual column
         *
         * @param {Object} column
         * @returns {void}
         */
        _disableAggregation: function (column) {
            column.aggregationOptions = true;
            column.isAggregation = ko.observable(false);
            column.aggregation.isActive = ko.observable(false);
            column.isAggregationEnabled = ko.observable(false);
            column.isFooterActive = ko.observable(false);
        },

        /**
         * Set disabled 'true' or 'false' to the available columns list based on 'state'
         * Toggle addingColumn functionality
         *
         * @param {Boolean} state
         * @returns {void}
         */
        toggleColumnsAvailability: function (state) {
            helpers.resolveComponent('entitiesList', this).addingColumn(!state);

            helpers.resolveComponent('chosenOptions', this).elems().forEach(function (element) {
                element.isDisabled(!state);
            });
        }
    };

    return function (target) {
        return target.extend(mixin);
    };
});
