/**
 * Validation Object Constructor
 */

define([
    'underscore',
    'mage/translate'
], function (_, $t) {
    'use strict';

    function Validator(constants) {
        this.constants = _.extend({
            regex: {
                text: /^[\s\a-zA-Z0-9!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?]*$/g, // eslint-disable-line no-useless-escape
                number: /[e+-]/gi
            },
            invalidChars: [
                '-',
                '+',
                'e',
                'E',
                'Shift'
            ]
        }, constants);

        this.classes = {
            valid: '-valid'
        };

        this.messages = {
            columnValidation: $t('Drag the column of type') + ' <b>{data-types}</b> ' + $t('from the left widget')
        };

        /**
         * Returns a message string for columns validation with the applicable data types
         *
         * @public
         * @param {Object} expression
         * @returns {String}
         */
        this.getValidationMessage = function (expression) {
            var dataTypes = this.constants.categories[this.getDataType(expression)]['data-types'];

            return this.messages.columnValidation.replace('{data-types}', dataTypes.join(', '));
        };

        /**
         * types: e.g. ('default', 'eav', 'foreign', 'custom')
         *
         * @public
         * @param {Object} column
         * @returns {Boolean}
         */
        this.isValidColumnType = function (column) {
            return _.contains(this.constants.columnTypes, column.column_type);
        };

        /**
         * data types: e.g. ('string', 'decimal', 'int')
         *
         * @public
         * @param {Object} expression - virtualColumn or brackets
         * @param {Object} column
         * @returns {Boolean}
         */
        this.isValidColumnDataType = function (expression, column) {
            return _.contains(this.constants.categories[expression.dataType()]['data-types'], column.type);
        };

        /**
         * Set a 'valid' css class to an element if column is pass the static tests
         *
         * @public
         * @param {Object} expression - virtualColumn or brackets
         * @param {Object} column
         * @param {jQuery} element - e.g. dropZone
         * @returns {Boolean}
         */
        this.setValidColumnClass = function (expression, column, element) {
            if (this.validateColumn(expression, column)) {
                element.addClass(this.classes.valid);
            }
        };

        /**
         * Validate current column by type and dataType
         *
         * @public
         * @param {Object} expression - virtualColumn or brackets
         * @param {Object} column
         * @returns {Boolean}
         */
        this.validateColumn = function (expression, column) {
            return this.isValidColumnType(column) && this.isValidColumnDataType(expression, column);
        };

        /**
         * @public
         * @param {String} type - 'text' or 'number'
         * @returns {Object} RegEx object
         */
        this.getRegex = function (type) {
            return this.constants.regex[type];
        };

        /**
         * Check an expression operands types and return expression type math based on the applicable data types
         *
         * @param {Object} expression - current virtual column
         * @returns {void}
         */
        this.changeTypeListener = function (expression) {
            var self = this;

            expression.context.subscribe(function (item) {
                item.forEach(function (element) {
                    element.type.subscribe(function () {
                        self.setDataType(expression, self.matchDataType(expression));
                    });
                });
            });
        };

        /**
         * Check an expression operands types and return expression type math based on the applicable data types
         *
         * @param {Object} element - current virtual column
         * @returns {String}
         */
        this.matchDataType = function (element) {
            var operandsTypesArray = [],
                categories = this.constants.categories,
                match;

            _.forEach(_.isUndefined(element.context) ? element.elements() : element.context(), function (item) {
                operandsTypesArray.push(item.type());
            });

            match = _.contains(operandsTypesArray, categories.arithmetic.identify)
                || this._findMatch(operandsTypesArray, categories.arithmetic['data-types']);

            return match ? categories.arithmetic.identify : categories.string.identify;
        };

        /**
         * @public
         * @param {Object} expression - current expression
         * @returns {String}
         */
        this.getDataType = function (expression) {
            return _.isUndefined(expression.dataType) ? expression.type() : expression.dataType();
        };

        /**
         * @public
         * @param {Object} expression - current expression
         * @param {String} type - expression data type
         * @returns {void}
         */
        this.setDataType = function (expression, type) {
            expression[_.isUndefined(expression.dataType) ? 'type' : 'dataType'](type);
        };

        /**
         * @public
         * @param {Object} expression - current expression
         * @param {String} type - input type
         * @returns {Boolean}
         */
        this.validateInputType = function (expression, type) {
            return _.contains(this.constants.categories[this.getDataType(expression)]['data-types'], type);
        };

        /**
         *  Determine if an array contains one or more items from another array.
         *
         * @private
         * @param {Array} haystack - the array to search.
         * @param {Array} arr - the array providing items to check for in the haystack.
         * @returns {Boolean}
         */
        this._findMatch = function (haystack, arr) {
            return arr.some(function (element) {
                return haystack.indexOf(element) >= 0;
            });
        };
    }

    return {
        init: function (constants) {
            return new Validator(constants);
        }
    };
});
