/**
 * Constructor operations popup component
 */

define([
    'jquery',
    'underscore',
    'ko',
    'uiComponent',
    'amrepbuilder_helpers',
    'uiRegistry',
    'constructor_validator'
], function ($, _, ko, Component, helpers, registry, Validator) {
    'use strict';

    return Component.extend({
        defaults: {
            template: 'Amasty_ReportBuilderVirtual/builder/constructor/operators_popup/wrapper',
            templates: {
                category: 'Amasty_ReportBuilderVirtual/builder/constructor/operators_popup/category',
                categoryItem: 'Amasty_ReportBuilderVirtual/builder/constructor/operators_popup/category-item',
                button: 'Amasty_ReportBuilder/components/button',
                removeIcon: 'Amasty_ReportBuilder/components/icons/remove',
                operatorsIconPath: 'Amasty_ReportBuilderVirtual/components/icons/operators/'
            },
            components: [
                'index = chosen_options',
                'index = chosen_column_toolbar',
                'index = amasty_report_builder',
                'index = component_constructor'
            ],
            categories: {},
            operators: {},
            operatorTypes: {
                UNARY: 'unary',
                BINARY: 'binary'
            }
        },
        scrollEventName: 'scroll.amrbvirtual',

        /**
         * Init observable variables
         *
         * @return {Object}
         */
        initObservable: function () {
            this._super()
                .observe({
                    isVisible: false,
                    positionTop: 0,
                    positionLeft: 0,
                    currentOperand: {}
                }).observe('categories');

            return this;
        },

        /**
         * Invokes initialize method of parent class,
         * contains initialization logic
         * @returns {void}
         */
        initialize: function () {
            var self = this;

            self._super();

            registry.get(self.components, function () {
                helpers.initComponentsArray(arguments, self);
            });

            this.initValidator({ categories: this.categories() });
            this._setCategoriesVisibility();
        },

        /**
         * Initialize Validator
         *
         * @param {Object} constants
         * @returns {void}
         */
        initValidator: function (constants) {
            this.validator = Validator.init(constants);
        },

        /**
         * Set operator to the current operand operation attribute, show toolbar, close popup
         *
         * @param {Object} item
         * @returns {void}
         */
        setOperator: function (item) {
            var currentOperand = this.currentOperand(),
                latestOperation = currentOperand.operation(),
                emptyOperation = this.constructor.operandTypes.NO_OPERATION,
                bracket;

            this._setOperation(item, currentOperand);

            if (!_.isUndefined(currentOperand.parentItemId)) {
                bracket = this.constructor.getParentItem(this.operandData.virtualColumn, currentOperand.parentItemId);

                this.validator.setDataType(bracket, item.category);
            }

            this.validator.setDataType(this.operandData.virtualColumn, item.category);

            if (latestOperation === emptyOperation && this.operandData.item.operation() !== emptyOperation) {
                this.constructor.showToolbar(this.operandData.virtualColumn, this.operandData.item);
            }

            this.closePopup();
        },

        /**
         * Save Operand data to operandData attribute
         *
         * @param {Object} virtualColumn
         * @param {Object} item - operand
         * @returns {void}
         */
        setOperandData: function (virtualColumn, item) {
            this.operandData = { virtualColumn: virtualColumn, item: item };
        },

        /**
         * Toggle operators popup visibility, set currentOperand data,  add scroll event, set popup position
         *
         * @param {Object} element - DOM node
         * @param {Object} data - contains operand (data.item)
         * @returns {void}
         */
        togglePopup: function (element, data) {
            var state = !this.isVisible();

            this._setCategoriesVisibility(data.item.type());
            this.currentOperand(data.item);
            this.isVisible(state);
            this._scrollEvent(element, state);
            this._setPosition(element);
        },

        /**
         * Close operators popup, purge currentOperand attribute, remove scroll listener
         * @returns {void}
         */
        closePopup: function () {
            this.currentOperand({});
            this.isVisible(false);
            this._scrollEvent(null, false);
        },

        /**
         * Returns an array of operators for specific category
         *
         * @param {String} category - category type
         * @returns {Array}
         */
        getOperatorsByCategory: function (category) {
            var result = [];

            Object.entries(this.operators).forEach(function ([key, value]) {
                if (value['category'] === category) {
                    result.push(value);
                }
            });

            return result;
        },

        /**
         * @param {Object} operator
         * @returns {void}
         */
        clearModifier: function (operator) {
            operator.modifier([]);
        },

        /**
         * @param {Object} operator
         * @returns {Boolean}
         */
        isOperatorUnary: function (operator) {
            return operator.type === this.operatorTypes.UNARY;
        },

        /**
         * @param {Object} item
         * @param {Object} currentOperand
         * @returns {void}
         */
        _setOperation: function (item, currentOperand) {
            if (this.isOperatorUnary(item)) {
                currentOperand.modifier([item.identity]);
            } else {
                currentOperand.operation(item.identity);
            }
        },

        /**
         * Init visibility observer for the operators categories or set their visibility based on an operand type
         *
         * @param {String} [type] - operand/constructor type
         * @returns {void}
         */
        _setCategoriesVisibility: function (type) {
            Object.entries(this.categories()).forEach(function ([key, item]) {
                if (ko.isObservable(item.isVisible) && !_.isUndefined(type)) {
                    item.isVisible(type === item.identify || _.contains(item['data-types'], type));
                } else {
                    item.isVisible = ko.observable(true);
                }
            });
        },

        /**
         * Set the operators popup css position props
         *
         * @param {Object} element - DOM node
         * @returns {void}
         */
        _setPosition: function (element) {
            this.positionTop(this._getPosition(element).top);
            this.positionLeft(this._getPosition(element).left);
        },

        /**
         * Get element top and left viewport relative offset
         *
         * @param {Object} element - DOM node
         * @return {Object}
         */
        _getPosition: function (element) {
            var rect = element.getBoundingClientRect();

            return {
                top: rect.bottom,
                left: rect.left + rect.width / 2
            };
        },

        /**
         * Scroll listener event that set popup position.
         *
         * @param {Object} element - DOM node
         * @param {Boolean} state - set 'false' to remove listener, 'true' to add listener
         * @returns {void}
         */
        _scrollEvent: function (element, state) {
            var self = this,
                ticking = false;

            if (state) {
                $(document).off(self.scrollEventName).on(self.scrollEventName, function () {
                    if (!ticking) {
                        window.requestAnimationFrame(function () {
                            self._setPosition(element);

                            ticking = false;
                        });

                        ticking = true;
                    }
                });
            } else {
                $(document).off(self.scrollEventName);
            }
        }
    });
});
