/**
 * Virtual Constructor wrapper
 */

define([
    'jquery',
    'underscore',
    'ko',
    'uiComponent',
    'amrepbuilder_helpers',
    'uiRegistry',
    'constructor_validator',
    'amrbvirtual_element_factory'
], function ($, _, ko, Component, helpers, registry, Validator, elementFactory) {
    'use strict';

    return Component.extend({
        defaults: {
            template: 'Amasty_ReportBuilderVirtual/builder/constructor/wrapper',
            templates: {
                button: 'Amasty_ReportBuilder/components/button',
                filterToolbar: 'Amasty_ReportBuilder/builder/column/filters/toolbar',
                constructorContainer: 'Amasty_ReportBuilderVirtual/builder/constructor/constructor-container',
                actionsToolbar: 'Amasty_ReportBuilderVirtual/builder/constructor/actions-toolbar',
                removeIcon: 'Amasty_ReportBuilder/components/icons/remove',
                operatorsIconPath: 'Amasty_ReportBuilderVirtual/components/icons/operators/',
                unaryOperatorContainer:
                    'Amasty_ReportBuilderVirtual/builder/constructor/operators_popup/unary-operator-container',
                operands: {
                    wrapper: 'Amasty_ReportBuilderVirtual/builder/constructor/operands/wrapper',
                    brackets: 'Amasty_ReportBuilderVirtual/builder/constructor/operands/brackets',
                    input: 'Amasty_ReportBuilderVirtual/builder/constructor/operands/input',
                    column: {
                        wrapper: 'Amasty_ReportBuilderVirtual/builder/constructor/operands/column/wrapper',
                        item: 'Amasty_ReportBuilderVirtual/builder/constructor/operands/column/item'
                    }
                }
            },
            components: [
                'index = chosen_options',
                'index = chosen_column_toolbar',
                'index = amasty_report_builder',
                'index = operators_popup'
            ],
            inputMaxLength: 64,
            imports: {
                categories: 'index = operators_popup:categories',
                operators: 'index = operators_popup:operators',
                isDragging: 'index = entities_list:isDragging'
            }
        },
        classes: {
            placeholder: 'amrbvirtual-dnd-placeholder'
        },
        selectors: {
            column: '[data-amrepbuilder-js="column"]:not(.-disabled)'
        },

        /**
         * Init observable variables
         *
         * @return {Object}
         */
        initObservable: function () {
            this._super()
                .observe({
                    isDropped: false,
                    isValid: false,
                    isButtonDisabled: false
                }).observe([
                    'categories',
                    'operators',
                    'isDragging'
                ]);

            return this;
        },

        /**
         * Invokes initialize method of parent class,
         * contains initialization logic
         * @returns {void}
         */
        initialize: function () {
            var self = this;

            self._super();

            registry.get(self.components, function () {
                helpers.initComponentsArray(arguments, self);

                self._setElementFactoryConstants();
                self.initValidator({
                    categories: self.categories(),
                    columnTypes: Object.keys(self.columnTypes)
                });
            });
        },

        /**
         * Initialize Validator
         *
         * @param {Object} constants
         * @returns {void}
         */
        initValidator: function (constants) {
            this.validator = Validator.init(constants);
        },

        /**
         * Initialize DropZone
         *
         * @param {Object} element
         * @param {VirtualContextElement} item
         * @param {Object} virtualColumn
         * @returns {void}
         */
        initDropZone: function (element, item, virtualColumn) {
            var self = this,
                dropZone = $(element);

            dropZone.droppable({
                activate: function (event, ui) {
                    self.currentItemData = $(ui.draggable).data('item');
                    self.currentDropItem = ui.draggable;
                },
                accept: function (node) {
                    var itemData = $(node).data('item');

                    self.isValid(false);

                    if (!_.isUndefined(itemData)) {
                        self.currentItemData = itemData;
                    }

                    if (!_.isUndefined(self.currentItemData)) {
                        self.isValid(self.validator.validateColumn(virtualColumn, self.currentItemData));
                    }

                    return self.isValid();
                },
                over: function () {
                    self._toggleDndHover(dropZone, true);
                },
                out: function () {
                    delete self.currentDropItem;

                    self._toggleDndHover(dropZone, false);
                },
                drop: function (event, ui) {
                    self.currentDropItem = ui.draggable;
                    self._toggleDndHover(dropZone, false);
                },
                deactivate: function () {
                    self._toggleDndHover(dropZone, false);
                }
            });
        },

        /**
         * Initialize Drag and Drop functionality for target column
         *
         * @param {Object} node prototype column node element
         * @param {Object} item operand element
         * @returns {void}
         */
        initDnD: function (node, item) {
            var self = this;

            $(node).sortable({
                items: self.selectors.column,
                placeholder: self.classes.placeholder,
                beforeStop: function (event, ui) {
                    $(ui.item).remove();
                },
                receive: function (event, ui) {
                    if (self.isValid()) {
                        self.addColumn($(ui.item).data('item'), item);
                    }
                }
            });
        },

        /**
         *  Add target column from available list to chosen list
         *
         *  @param {Object} prototype target prototype
         *  @param {Object} item operand element
         *  @returns {void}
         */
        addColumn: function (prototype, item) {
            item.columnData(prototype);
            this._setColumnOperandId(item);
            item.type(prototype.type);
            this.isDropped(true);
            this.isButtonDisabled(false);
        },

        /**
         * Add new operand to the constructor list
         *
         * @param {String} categoryType - type of category (parent operand)
         * @param {Object} item - operand
         * @param {Object} virtualColumn - root constructor element with an context array
         * @param {String} category - operand category
         * @returns {void}
         */
        addOperand: function (categoryType, item, virtualColumn, category) {
            this.isDropped(false);
            this.hideToolbar(virtualColumn, item, categoryType);
            this.isButtonDisabled(true);

            switch (categoryType) {
                case 'brackets':
                    this._setOperand(item.elements, category, item);
                    break;
                default:
                    this._setOperand(virtualColumn.context, category);
            }
        },

        /**
         * Remove  operand from the constructor list or from the brackets
         *
         * @param {Object} item - operand
         * @param {Object} virtualColumn - root constructor element with an context array
         * @returns {void}
         */
        removeOperand: function (item, virtualColumn) {
            var isRoot = _.isUndefined(item.parentItemId),
                parentItem = this.getParentItem(virtualColumn, item.parentItemId);

            if (isRoot) {
                this.processRemovingOperand(virtualColumn, item);
                this._resetOperator(virtualColumn.context());
                this.validator.setDataType(virtualColumn, this.validator.matchDataType(virtualColumn));
            } else {
                this.processRemovingOperand(parentItem, item);
                this._resetOperator(parentItem.elements());
                this.validator.setDataType(parentItem, this.validator.matchDataType(parentItem));
            }
        },

        /**
         * @param {Object} target - expression: virtual column or brackets
         * @param {Object} item - operand
         * @returns {void}
         */
        processRemovingOperand: function (target, item) {
            (_.isUndefined(target.context) ? target.elements : target.context).remove(function (element) {
                return element.id === item.id;
            });

            target.isShowToolbar(
                !(_.isUndefined(target.context) ? target.elements() : target.context()).length
            );
        },

        /**
         * Show "select an operation" button
         *
         * @param {Object} item - operand
         * @returns {Boolean}
         */
        isShowSelectOperationButton: function (item) {
            return !_.isEmpty(item.type());
        },

        /**
         * Hide constructor toolbar visibility
         *
         * @param {Object} virtualColumn
         * @param {Object} item - operand
         * @param {String} categoryType - operand type
         * @returns {void}
         */
        hideToolbar: function (virtualColumn, item, categoryType) {
            if (categoryType === this.operandTypes.BRACKETS) {
                item.isShowToolbar(false);
            } else {
                virtualColumn.isShowToolbar(false);
            }
        },

        /**
         * Show constructor toolbar visibility
         *
         * @param {Object} virtualColumn
         * @param {Object} item - operand
         * @returns {void}
         */
        showToolbar: function (virtualColumn, item) {
            if (item.parentItemId) {
                this.getParentItem(virtualColumn, item.parentItemId).isShowToolbar(true);
            } else {
                virtualColumn.isShowToolbar(true);
            }
        },

        /**
         * Get the parent bracket operand
         *
         * @param {Object} virtualColumn
         * @param {String} id - bracket id
         * @returns {Object}
         */
        getParentItem: function (virtualColumn, id) {
            return _.find(virtualColumn.context(), function (item) {
                return item.id === id;
            });
        },

        /**
         * Open/close operators popup
         *
         * @param {Object} element - DOM node
         * @param {Object} data
         * @param {Object} virtualColumn
         * @param {Object} item - operand
         * @returns {void}
         */
        togglePopup: function (element, data, virtualColumn, item) {
            this.operatorsPopup.setOperandData(virtualColumn, item);
            this.operatorsPopup.togglePopup(element, data);
        },

        /**
         * Close footer with constructor, remove builder overlay and enable columns sorting
         *
         * @param {Object} item
         * @returns {void}
         */
        close: function (item) {
            item.isFooterActive(false);
            item.isOpen(false);
            this.toolbar.toggleColumnsAvailability(false);
            this.chosenOptions.isOverlay(false);
            this.toolbar.toggleSorting(item);
            this.operatorsPopup.closePopup();
        },

        /**
         *
         * @param {Object} item - operand
         * @param {String} type - operand data type
         * @returns {void}
         */
        setOperandInputType: function (item, type) {
            switch (type) {
                case this.inputTypes.STRING:
                    item.htmlType('text');
                    break;
                case this.inputTypes.DECIMAL:
                default:
                    item.htmlType('number');
                    break;
            }

            this.isButtonDisabled(false);
            item.type(type);
        },

        /**
         * Returns a message for DnD column zone
         *
         * @public
         * @param {Object} expression
         * @returns {String}
         */
        getDnDMessage: function (expression) {
            return this.validator.getValidationMessage(expression);
        },

        /**
         * Check if input type isn't available as operand
         *
         * @public
         * @param {Object} expression
         * @param {String} type
         * @returns {Boolean}
         */
        inputTypeDisabled: function (expression, type) {
            return !this.validator.validateInputType(expression, type);
        },

        /**
         * @public
         * @param {Object} item - column
         * @returns {Boolean}
         */
        getColumnTitle: function (item) {
            return _.isFunction(item.title) ? item.title() : item.title;
        },

        /**
         * Callback after constructor is rendered
         *
         * @public
         * @param {Object} column - virtual column
         * @returns {Boolean}
         */
        updateAfterRender: function (column) {
            this._resetOperators(column.context());
        },

        /**
         * @param {Object} item - an operand
         * @returns {String}
         */
        getOperationName: function (item) {
            return _.isNull(item.operation()) ? 'settings' : item.operation();
        },

        /**
         * Set the 'settings operator' to the last operand in an expression
         *
         * @param {Array} elements - array of operands
         * @returns {void}
         */
        _resetOperator: function (elements) {
            if (elements.length) {
                elements[elements.length - 1].operation(this.operandTypes.NO_OPERATION);
            }

            this.isButtonDisabled(false);
        },

        /**
         * @param {Array} column - virtual column
         * @returns {void}
         */
        _resetOperators: function (column) {
            this._resetOperator(column);

            _.each(column, function (item) {
                if (item.category === 'brackets') {
                    this._resetOperator(item.elements());
                }
            }.bind(this));
        },

        /**
         * @param {jQuery} element - dropZone
         * @param {Boolean} state
         * @returns {void}
         */
        _toggleDndHover: function (element, state) {
            element.toggleClass(this.chosenOptions.classes.hover, state);
        },

        /**
         * Set operand types constants to the constructor uiClass
         *
         * @returns {void}
         */
        _setElementFactoryConstants: function () {
            this.operandTypes = elementFactory.operandTypes;
            this.inputTypes = elementFactory.inputTypes;
        },

        /**
         *  Add the column_id attribute to a columnData of the chosen column
         *
         *  @param {VirtualContextElement} item operand element
         *  @returns {void}
         */
        _setColumnOperandId: function (item) {
            item.columnData().column_id = item.columnData().id;
        },

        /**
         * Set operand to the operands list
         *
         * @param {Object} target - target array of operands
         * @param {String} category - operand category
         * @param {Object} prototype - parent brackets object
         * @returns {void}
         */
        _setOperand: function (target, category, prototype) {
            target.push(this._getOperandObject(category, prototype));
        },

        /**
         * Return new operand object
         *
         * @param {String} category - type of operand
         * @param {Object} prototype - parent brackets object
         * @return {ObservableVirtualContextElement}
         */
        _getOperandObject: function (category, prototype) {
            return elementFactory.create(category, prototype);
        }
    });
});
