<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Report Builder Virtual Columns (System)
*/

declare(strict_types=1);

namespace Amasty\ReportBuilderVirtual\Test\Integration\Model\SelectResolver;

use Amasty\ReportBuilder\Api\Data\ReportColumnInterface;
use Amasty\ReportBuilder\Api\Data\SelectColumnInterface;
use Amasty\ReportBuilder\Model\EntityScheme\Column\AggregationType;
use Amasty\ReportBuilder\Model\SelectResolver\SelectColumn\SelectEavColumn;
use Amasty\ReportBuilderVirtual\Api\Data\ReportVirtualColumnInterface;
use Amasty\ReportBuilderVirtual\Model\Context\Element\Brackets;
use Amasty\ReportBuilderVirtual\Model\Context\Element\Column;
use Amasty\ReportBuilderVirtual\Model\Context\Element\Input;
use Amasty\ReportBuilderVirtual\Model\Operator\Binary\Addition;
use Amasty\ReportBuilderVirtual\Model\Operator\Binary\Concatenation;
use Amasty\ReportBuilderVirtual\Model\Operator\Binary\Division;
use Amasty\ReportBuilderVirtual\Model\Operator\Binary\Multiplication;
use Amasty\ReportBuilderVirtual\Model\Operator\Binary\SeparatedConcatenation;
use Amasty\ReportBuilderVirtual\Model\Operator\Binary\Subtraction;
use Amasty\ReportBuilderVirtual\Model\Operator\Unary\Absolute;
use Amasty\ReportBuilderVirtual\Model\Operator\Unary\SquareRoot;
use Amasty\ReportBuilderVirtual\Model\SelectResolver\SelectColumn\SelectVirtualColumn;
use Magento\TestFramework\Helper\Bootstrap;

/**
 * @magentoAppArea adminhtml
 * @magentoAppIsolation disabled
 * @magentoDbIsolation disabled
 * @magentoDataFixture Amasty_ReportBuilder::Test/_files/report.php
 */
class ColumnResolverTest extends \Amasty\ReportBuilder\Test\Integration\Model\SelectResolver\ColumnResolverTest
{
    /**
     * @return array[][]
     * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
     */
    public function dataProvider(): array
    {
        return [
            'virtual column' => [
                [
                    'order.grand_total' =>
                        [
                            ReportColumnInterface::COLUMN_ID => 'order.grand_total',
                            ReportColumnInterface::AGGREGATION_TYPE => AggregationType::TYPE_SUM,
                        ],
                    'virtual_1' => [
                        ReportColumnInterface::COLUMN_ID => 'virtual_1',
                        ReportColumnInterface::EXTENSION_ATTRIBUTES_KEY => [
                            'data' => [
                                ReportVirtualColumnInterface::EXTENDED_KEY => $this->createVirtual([
                                    ReportVirtualColumnInterface::CONTEXT => [
                                        [
                                            Column::CATEGORY_KEY => Column::CATEGORY,
                                            Column::COLUMN_ID => 'order_item.price',
                                            Column::TYPE_KEY => 'decimal',
                                            Column::ORDER_KEY => 0,
                                            Column::MODIFIER_KEY => null,
                                            Column::OPERATION_KEY => Multiplication::IDENTITY,
                                        ],
                                        [
                                            Brackets::CATEGORY_KEY => Brackets::CATEGORY,
                                            Brackets::TYPE_KEY => 'decimal',
                                            Brackets::ORDER_KEY => 1,
                                            Brackets::MODIFIER_KEY => null,
                                            Brackets::OPERATION_KEY => Division::IDENTITY,
                                            Brackets::ELEMENTS_KEY => [
                                                [
                                                    Column::CATEGORY_KEY => Column::CATEGORY,
                                                    Column::COLUMN_ID => 'order_item.qty_invoiced',
                                                    Column::TYPE_KEY => 'decimal',
                                                    Column::ORDER_KEY => 0,
                                                    Column::MODIFIER_KEY => null,
                                                    Column::OPERATION_KEY => Addition::IDENTITY,
                                                ],
                                                [
                                                    Column::CATEGORY_KEY => Column::CATEGORY,
                                                    Column::COLUMN_ID => 'order_item.qty_backordered',
                                                    Column::TYPE_KEY => 'decimal',
                                                    Column::ORDER_KEY => 1,
                                                    Column::MODIFIER_KEY => null,
                                                    Column::OPERATION_KEY => null,
                                                ],
                                            ]
                                        ],
                                        [
                                            Input::CATEGORY_KEY => Input::CATEGORY,
                                            Input::VALUE_KEY => 2,
                                            Input::TYPE_KEY => 'decimal',
                                            Input::ORDER_KEY => 2,
                                            Input::MODIFIER_KEY => [Absolute::IDENTITY],
                                            Input::OPERATION_KEY => Subtraction::IDENTITY,
                                        ],
                                        [
                                            Column::CATEGORY_KEY => Column::CATEGORY,
                                            Column::COLUMN_ID => 'catalog_product.price',
                                            Column::TYPE_KEY => 'decimal',
                                            Column::ORDER_KEY => 3,
                                            Column::MODIFIER_KEY => null,
                                            Column::OPERATION_KEY => null,
                                        ]
                                    ],
                                    ReportVirtualColumnInterface::DATA_TYPE => 'decimal'
                                ])
                            ]
                        ]
                    ]
                ],
                [
                    'order.grand_total' => [
                        SelectColumnInterface::ALIAS => 'order_grand_total',
                        SelectColumnInterface::EXPRESSION => 'order.grand_total',
                        SelectColumnInterface::ENTITY_NAME => 'order',
                        SelectColumnInterface::AGGREGATED_EXPRESSION => 'SUM(%s)',
                        SelectColumnInterface::USE_AGGREGATION => false,
                        SelectColumnInterface::COLUMN_ID => 'order.grand_total',
                        SelectColumnInterface::EXTERNAL_AGGREGATED_EXPRESSION => 'SUM(%s)'
                    ],
                    'virtual_1' => [
                        SelectColumnInterface::ALIAS => 'virtual_1',
                        SelectColumnInterface::EXPRESSION
                        => 'IFNULL(IFNULL(SUM(order_item_price), 0)' .
                            ' * IFNULL((IFNULL(SUM(order_item_qty_invoiced), 0)' .
                            ' + IFNULL(SUM(order_item_qty_backordered), 0)), 0)' .
                            ' / IF(ABS(IFNULL(2, 0)), ABS(IFNULL(2, 0)), 1), 0)' .
                            ' - IFNULL(SUM(catalog_product_price), 0)',
                        SelectColumnInterface::AGGREGATED_EXPRESSION => '%s',
                        SelectColumnInterface::USE_AGGREGATION => false,
                        SelectColumnInterface::COLUMN_ID => 'virtual_1',
                        SelectVirtualColumn::RELATED_COLUMNS => [
                            'order_item.price' => [
                                SelectColumnInterface::ALIAS => 'order_item_price',
                                SelectColumnInterface::EXPRESSION => 'order_item_price',
                                SelectColumnInterface::ENTITY_NAME => 'order_item',
                                SelectColumnInterface::AGGREGATED_EXPRESSION =>
                                    'SUM(%s)',
                                SelectColumnInterface::USE_AGGREGATION => true,
                                SelectColumnInterface::COLUMN_ID => 'order_item.price',
                                SelectColumnInterface::EXPRESSION_INTERNAL => 'order_item.price',
                                SelectColumnInterface::EXTERNAL_AGGREGATED_EXPRESSION => 'SUM(%s)'
                            ],
                            'order_item.qty_invoiced' => [
                                SelectColumnInterface::ALIAS => 'order_item_qty_invoiced',
                                SelectColumnInterface::EXPRESSION => 'order_item_qty_invoiced',
                                SelectColumnInterface::ENTITY_NAME => 'order_item',
                                SelectColumnInterface::AGGREGATED_EXPRESSION =>
                                    'SUM(%s)',
                                SelectColumnInterface::USE_AGGREGATION => true,
                                SelectColumnInterface::COLUMN_ID => 'order_item.qty_invoiced',
                                SelectColumnInterface::EXPRESSION_INTERNAL => 'order_item.qty_invoiced',
                                SelectColumnInterface::EXTERNAL_AGGREGATED_EXPRESSION => 'SUM(%s)'
                            ],
                            'order_item.qty_backordered' => [
                                SelectColumnInterface::ALIAS => 'order_item_qty_backordered',
                                SelectColumnInterface::EXPRESSION => 'order_item_qty_backordered',
                                SelectColumnInterface::ENTITY_NAME => 'order_item',
                                SelectColumnInterface::AGGREGATED_EXPRESSION =>
                                    'SUM(%s)',
                                SelectColumnInterface::USE_AGGREGATION => true,
                                SelectColumnInterface::COLUMN_ID => 'order_item.qty_backordered',
                                SelectColumnInterface::EXPRESSION_INTERNAL => 'order_item.qty_backordered',
                                SelectColumnInterface::EXTERNAL_AGGREGATED_EXPRESSION => 'SUM(%s)'
                            ],
                            'catalog_product.price' => [
                                SelectColumnInterface::ALIAS => 'catalog_product_price',
                                SelectColumnInterface::EXPRESSION => 'catalog_product_price',
                                SelectColumnInterface::ENTITY_NAME => 'catalog_product',
                                SelectColumnInterface::AGGREGATED_EXPRESSION =>
                                    'SUM(%s)',
                                SelectColumnInterface::USE_AGGREGATION => true,
                                SelectColumnInterface::COLUMN_ID => 'catalog_product.price',
                                SelectEavColumn::ATTRIBUTE_ID => $this->getPriceId(),
                                SelectColumnInterface::EXPRESSION_INTERNAL => 'catalog_product.price',
                                SelectColumnInterface::EXTERNAL_AGGREGATED_EXPRESSION => 'SUM(%s)'
                            ],
                        ]
                    ],
                ]
            ],
            'virtual column same as global' => [
                [
                    'order.entity_id' => [
                        ReportColumnInterface::COLUMN_ID => 'order.entity_id',
                        ReportColumnInterface::AGGREGATION_TYPE => AggregationType::TYPE_COUNT,
                        ReportColumnInterface::FILTER => '{"from":500,"to":""}',
                        ReportColumnInterface::VISIBILITY => true,
                        ReportColumnInterface::POSITION => 1,
                        ReportColumnInterface::CUSTOM_TITLE => 'Custom title',
                    ],
                    'virtual_1' => [
                        ReportColumnInterface::COLUMN_ID => 'virtual_1',
                        ReportColumnInterface::EXTENSION_ATTRIBUTES_KEY => [
                            'data' => [
                                ReportVirtualColumnInterface::EXTENDED_KEY => $this->createVirtual([
                                    ReportVirtualColumnInterface::CONTEXT => [
                                        [
                                            Column::CATEGORY_KEY => Column::CATEGORY,
                                            Column::COLUMN_ID => 'order.entity_id',
                                            ReportColumnInterface::AGGREGATION_TYPE => AggregationType::TYPE_SUM,
                                            Column::TYPE_KEY => 'int',
                                            Column::ORDER_KEY => 0,
                                            Column::MODIFIER_KEY => [Absolute::IDENTITY],
                                            Column::OPERATION_KEY => Division::IDENTITY,
                                        ],
                                        [
                                            Input::CATEGORY_KEY => Input::CATEGORY,
                                            Input::VALUE_KEY => 4,
                                            Input::TYPE_KEY => 'int',
                                            Input::ORDER_KEY => 1,
                                            Input::MODIFIER_KEY => [SquareRoot::IDENTITY, Absolute::IDENTITY],
                                            Input::OPERATION_KEY => null,
                                        ],
                                    ],
                                    ReportVirtualColumnInterface::DATA_TYPE => 'int'
                                ])
                            ]
                        ]
                    ]
                ],
                [
                    'order.entity_id' => [
                        SelectColumnInterface::ALIAS => 'order_entity_id',
                        SelectColumnInterface::EXPRESSION => 'order.entity_id',
                        SelectColumnInterface::ENTITY_NAME => 'order',
                        SelectColumnInterface::AGGREGATED_EXPRESSION => 'COUNT(DISTINCT %s)',
                        SelectColumnInterface::USE_AGGREGATION => false,
                        SelectColumnInterface::COLUMN_ID => 'order.entity_id',
                        SelectColumnInterface::EXTERNAL_AGGREGATED_EXPRESSION => 'SUM(%s)'
                    ],
                    'virtual_1' => [
                        SelectColumnInterface::ALIAS => 'virtual_1',
                        SelectColumnInterface::EXPRESSION =>
                            'ABS(IFNULL(order.entity_id, 0)) ' .
                            '/ IF(ABS(IFNULL(SQRT(IFNULL(4, 0)), 0)), ABS(IFNULL(SQRT(IFNULL(4, 0)), 0)), 1)',
                        SelectColumnInterface::AGGREGATED_EXPRESSION => '%s',
                        SelectColumnInterface::USE_AGGREGATION => false,
                        SelectColumnInterface::COLUMN_ID => 'virtual_1',
                        SelectVirtualColumn::RELATED_COLUMNS => [
                            'order.entity_id' => [
                                SelectColumnInterface::ALIAS => 'order_entity_id',
                                SelectColumnInterface::EXPRESSION => 'order.entity_id',
                                SelectColumnInterface::ENTITY_NAME => 'order',
                                SelectColumnInterface::AGGREGATED_EXPRESSION => 'SUM(%s)',
                                SelectColumnInterface::USE_AGGREGATION => false,
                                SelectColumnInterface::COLUMN_ID => 'order.entity_id',
                                SelectColumnInterface::EXTERNAL_AGGREGATED_EXPRESSION => 'SUM(%s)'
                            ]
                        ]
                    ]
                ]
            ],
            'virtual column String type' => [
                [
                    'virtual_3' => [
                        ReportColumnInterface::COLUMN_ID => 'virtual_3',
                        ReportColumnInterface::EXTENSION_ATTRIBUTES_KEY => [
                            'data' => [
                                ReportVirtualColumnInterface::EXTENDED_KEY => $this->createVirtual([
                                    ReportVirtualColumnInterface::CONTEXT => [
                                        [
                                            Column::CATEGORY_KEY => Column::CATEGORY,
                                            Column::COLUMN_ID => 'catalog_product.sku',
                                            ReportColumnInterface::AGGREGATION_TYPE => AggregationType::TYPE_NONE,
                                            Column::TYPE_KEY => 'varchar',
                                            Column::ORDER_KEY => 0,
                                            Column::MODIFIER_KEY => null,
                                            Column::OPERATION_KEY => Concatenation::IDENTITY,
                                        ],
                                        [
                                            Input::CATEGORY_KEY => Input::CATEGORY,
                                            Input::VALUE_KEY => ' SKU',
                                            Input::TYPE_KEY => 'varchar',
                                            Input::ORDER_KEY => 1,
                                            Input::MODIFIER_KEY => null,
                                            Input::OPERATION_KEY => SeparatedConcatenation::IDENTITY,
                                        ],
                                        [
                                            Input::CATEGORY_KEY => Input::CATEGORY,
                                            Input::VALUE_KEY => ' SKU',
                                            Input::TYPE_KEY => 'varchar',
                                            Input::ORDER_KEY => 2,
                                            Input::MODIFIER_KEY => null,
                                            Input::OPERATION_KEY => null,
                                        ],
                                    ],
                                    ReportVirtualColumnInterface::DATA_TYPE => 'varchar'
                                ])
                            ]
                        ]
                    ]
                ],
                [
                    'virtual_3' => [
                        SelectColumnInterface::ALIAS => 'virtual_3',
                        SelectColumnInterface::EXPRESSION =>
                            "CONCAT(COALESCE(" .
                            "CONCAT(COALESCE(catalog_product_sku, ''), ' ', COALESCE(' SKU', '')), '')," .
                            " ', ', " .
                            "COALESCE(' SKU', ''))",
                        SelectColumnInterface::AGGREGATED_EXPRESSION => '%s',
                        SelectColumnInterface::USE_AGGREGATION => false,
                        SelectColumnInterface::COLUMN_ID => 'virtual_3',
                        SelectVirtualColumn::RELATED_COLUMNS => [
                            'catalog_product.sku' => [
                                SelectColumnInterface::ALIAS => 'catalog_product_sku',
                                SelectColumnInterface::EXPRESSION => 'catalog_product_sku',
                                SelectColumnInterface::EXPRESSION_INTERNAL => 'catalog_product.sku',
                                SelectColumnInterface::ENTITY_NAME => 'catalog_product',
                                SelectColumnInterface::AGGREGATED_EXPRESSION => '%s',
                                SelectColumnInterface::USE_AGGREGATION => true,
                                SelectColumnInterface::COLUMN_ID => 'catalog_product.sku',
                                SelectColumnInterface::EXTERNAL_AGGREGATED_EXPRESSION => '%s'
                            ]
                        ]
                    ]
                ]
            ]
        ];
    }

    private function createVirtual(array $data): ReportVirtualColumnInterface
    {
        return Bootstrap::getObjectManager()->create(ReportVirtualColumnInterface::class, ['data' => $data]);
    }
}
