<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Report Builder Virtual Columns (System)
*/

declare(strict_types=1);

namespace Amasty\ReportBuilderVirtual\Test\Integration\Model\Operator;

use Amasty\ReportBuilderVirtual\Model\Operator\Binary\Addition;
use Amasty\ReportBuilderVirtual\Model\Operator\Binary\Subtraction;
use Amasty\ReportBuilderVirtual\Model\Operator\OperatorProcessor;
use Amasty\ReportBuilderVirtual\Model\Operator\OperatorRepository;
use Magento\TestFramework\Helper\Bootstrap;
use PHPUnit\Framework\TestCase;

/**
 * @magentoAppArea adminhtml
 * @magentoAppIsolation disabled
 * @magentoDbIsolation disabled
 */
class OperatorProcessorTest extends TestCase
{
    /**
     * @var OperatorProcessor
     */
    private $model;

    /**
     * @var OperatorRepository
     */
    private $operatorRepository;

    protected function setUp(): void
    {
        parent::setUp();

        $this->model = Bootstrap::getObjectManager()->get(OperatorProcessor::class);
        $this->operatorRepository = Bootstrap::getObjectManager()->get(OperatorRepository::class);
    }

    /**
     * @covers OperatorProcessor::buildExpression
     *
     * @dataProvider dataProvider
     *
     * @param string $operatorIdentity
     * @param string[]|int[]|float[] $expressions
     * @param string $expectedResult
     */
    public function testBuildExpression(string $operatorIdentity, array $expressions, string $expectedResult): void
    {
        $result = $this->model->buildExpression($operatorIdentity, ...$expressions);

        self::assertSame($expectedResult, $result);
    }

    public function testIsAllOperatorsTested(): void
    {
        $notTestedOperators = $this->operatorRepository->getList();
        foreach ($this->dataProvider() as $case) {
            $identity = $case[0];
            unset($notTestedOperators[$identity]);
        }

        if (!empty($notTestedOperators)) {
            self::fail(
                'All operators should have test case operators without tests: '
                . implode(', ', array_keys($notTestedOperators))
            );
        }
    }

    public function dataProvider(): array
    {
        return [
            'plus table' => [
                Addition::IDENTITY,
                ['`table.column`', '`table.column2`'],
                'IFNULL(`table.column`, 0) + IFNULL(`table.column2`, 0)'
            ],
            'minus int' => [
                Subtraction::IDENTITY,
                [2, 3],
                'IFNULL(2, 0) - IFNULL(3, 0)'
            ],
            'multiply float' => [
                \Amasty\ReportBuilderVirtual\Model\Operator\Binary\Multiplication::IDENTITY,
                [2.5, 3.0000001],
                'IFNULL(2.5, 0) * IFNULL(3.0000001, 0)'
            ],
            'Division float' => [
                \Amasty\ReportBuilderVirtual\Model\Operator\Binary\Division::IDENTITY,
                [1, 0.00001],
                '1 / IF(1.0E-5, 1.0E-5, 1)'
            ],
            'SeparatedConcatenation' => [
                \Amasty\ReportBuilderVirtual\Model\Operator\Binary\SeparatedConcatenation::IDENTITY,
                ['a', 'b'],
                "CONCAT(COALESCE(a, ''), ', ', COALESCE(b, ''))"
            ],
            'Concatenation' => [
                \Amasty\ReportBuilderVirtual\Model\Operator\Binary\Concatenation::IDENTITY,
                ['"a"', '"b"'],
                'CONCAT(COALESCE("a", \'\'), \' \', COALESCE("b", \'\'))'
            ],
            'Absolute float' => [
                \Amasty\ReportBuilderVirtual\Model\Operator\Unary\Absolute::IDENTITY,
                [3.00001],
                'ABS(IFNULL(3.00001, 0))'
            ],
            'SquareRoot' => [
                \Amasty\ReportBuilderVirtual\Model\Operator\Unary\SquareRoot::IDENTITY,
                ['table.column'],
                'SQRT(IFNULL(table.column, 0))'
            ],
            'Exponentiation' => [
                \Amasty\ReportBuilderVirtual\Model\Operator\Unary\Exponentiation::IDENTITY,
                ['table.column'],
                'POWER(IFNULL(table.column, 0), 2)'
            ],
        ];
    }
}
