<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Report Builder Virtual Columns (System)
*/

declare(strict_types=1);

namespace Amasty\ReportBuilderVirtual\Model\SelectResolver\SelectColumn\Adapter;

use Amasty\ReportBuilder\Api\ColumnInterface;
use Amasty\ReportBuilder\Api\Data\ReportColumnInterface;
use Amasty\ReportBuilder\Api\Data\SelectColumnInterface;
use Amasty\ReportBuilder\Exception\NotExistColumnException;
use Amasty\ReportBuilder\Model\EntityScheme\Column\AggregationType;
use Amasty\ReportBuilder\Model\Report\ColumnsResolver;
use Amasty\ReportBuilder\Model\SelectResolver\SelectColumn\Adapter\ColumnAdapterInterface;
use Amasty\ReportBuilder\Model\SelectResolver\SelectColumn\ColumnBuilder;
use Amasty\ReportBuilder\Model\SelectResolver\SelectColumn\ColumnStorageInterface;
use Amasty\ReportBuilderVirtual\Model\Context\ContextListProvider;
use Amasty\ReportBuilderVirtual\Model\ContextProcessor\VirtualExpressionProcessor;
use Amasty\ReportBuilderVirtual\Model\SelectResolver\SelectColumn\SelectVirtualColumn;
use Amasty\ReportBuilderVirtual\Model\SelectResolver\SelectColumn\SelectVirtualColumnFactory;
use Amasty\ReportBuilder\Model\SelectResolver\SelectColumn\ColumnStorageInterfaceFactory;

class VirtualAdapter implements ColumnAdapterInterface
{
    /**
     * @var SelectVirtualColumnFactory
     */
    private $columnFactory;

    /**
     * @var ColumnBuilder|null
     */
    private $contextColumnBuilder;

    /**
     * @var ColumnStorageInterfaceFactory
     */
    private $storageFactory;

    /**
     * @var ContextListProvider
     */
    private $contextProvider;

    /**
     * @var VirtualExpressionProcessor
     */
    private $expressionProcessor;

    /**
     * @var ColumnsResolver
     */
    private $columnsResolver;

    public function __construct(
        SelectVirtualColumnFactory $columnFactory,
        ColumnStorageInterfaceFactory $storageFactory,
        ContextListProvider $contextProvider,
        VirtualExpressionProcessor $expressionProcessor,
        ColumnsResolver $columnsResolver,
        ColumnBuilder $contextColumnBuilder = null
    ) {
        $this->columnFactory = $columnFactory;
        $this->contextColumnBuilder = $contextColumnBuilder;
        $this->storageFactory = $storageFactory;
        $this->contextProvider = $contextProvider;
        $this->expressionProcessor = $expressionProcessor;
        $this->columnsResolver = $columnsResolver;
    }

    /**
     * @param ReportColumnInterface $reportColumn
     * @param ColumnInterface|null $schemeColumn
     *
     * @return SelectColumnInterface|null
     */
    public function process(
        ReportColumnInterface $reportColumn,
        ?ColumnInterface $schemeColumn
    ): SelectColumnInterface {
        /** @var SelectVirtualColumn $selectColumn */
        $selectColumn = $this->columnFactory->create();

        $virtualColumn = $reportColumn->getExtensionAttributes()->getVirtualColumn();

        $context = $this->contextProvider->getByColumn($virtualColumn);
        /** @var ColumnStorageInterface $contextColumnsStorage */
        $contextColumnsStorage = $this->storageFactory->create();

        $this->contextColumnBuilder->build($context->getColumns(), $contextColumnsStorage);
        $columnId = $reportColumn->getColumnId();

        $selectColumn->setColumnId($columnId);
        $selectColumn->setAlias($columnId);
        $selectColumn->setIsUseAggregation(false);
        $selectColumn->setRelatedColumns($contextColumnsStorage->getAllColumns());
        $selectColumn->setAggregatedExpression(AggregationType::EXPRESSION_NONE);

        $expression = $this->expressionProcessor->execute($virtualColumn, $selectColumn);
        $selectColumn->setExpression($expression);

        return $selectColumn;
    }

    /**
     * @param ReportColumnInterface $reportColumn
     * @param ColumnInterface|null $schemeColumn
     *
     * @return bool
     */
    public function isApplicable(ReportColumnInterface $reportColumn, ?ColumnInterface $schemeColumn): bool
    {
        return $schemeColumn === null && $reportColumn->getExtensionAttributes()->getVirtualColumn() !== null;
    }

    /**
     * Validate schema columns which is included into virtual column.
     *
     * @throws NotExistColumnException
     */
    public function validate(string $columnId): void
    {
        $reportColumns = $this->columnsResolver->getReportColumns();
        if (!isset($reportColumns[$columnId])) {
            throw new NotExistColumnException(__('Virtual column is not found for validation'));
        }

        $reportColumn = $reportColumns[$columnId];

        $virtualColumn = $reportColumn->getExtensionAttributes()->getVirtualColumn();

        $context = $this->contextProvider->getByColumn($virtualColumn);
        foreach ($context->getColumns() as $column) {
            $this->contextColumnBuilder->validateColumn($column);
        }
    }
}
