<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Report Builder Virtual Columns (System)
*/

declare(strict_types=1);

namespace Amasty\ReportBuilderVirtual\Model\Operator;

use Magento\Framework\Exception\AlreadyExistsException;

class OperatorRepository
{
    /**
     * @var OperatorInterface[]
     */
    private $operatorProcessors;

    /**
     * @param OperatorInterface[] $operatorProcessors
     */
    public function __construct(array $operatorProcessors)
    {
        $this->initProcessorsStorage($operatorProcessors);
    }

    /**
     * @param OperatorInterface[] $processors
     *
     * @throws AlreadyExistsException
     */
    private function initProcessorsStorage(array $processors): void
    {
        foreach ($processors as $operator) {
            $identity = $operator->getIdentity();
            if (isset($this->operatorProcessors[$identity])) {
                throw new AlreadyExistsException(
                    __('Operator with the same ID "%id" already set', ['id' => $identity])
                );
            }

            $this->operatorProcessors[$identity] = $operator;
        }
    }

    /**
     * @param string $identity
     *
     * @return OperatorInterface
     */
    public function get(string $identity): OperatorInterface
    {
        return $this->operatorProcessors[$identity];
    }

    /**
     * Is provided operator exist in repository?
     *
     * @param string $identity
     *
     * @return bool
     */
    public function isExist(string $identity): bool
    {
        return isset($this->operatorProcessors[$identity]);
    }

    /**
     * @return OperatorInterface[]
     */
    public function getList(): array
    {
        return $this->operatorProcessors;
    }
}
