<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Report Builder Virtual Columns (System)
*/

declare(strict_types=1);

namespace Amasty\ReportBuilderVirtual\Model\Context;

use Amasty\ReportBuilderVirtual\Exceptions\ContextValidationException;
use Amasty\ReportBuilderVirtual\Model\Context\Element\Brackets;
use Amasty\ReportBuilderVirtual\Model\Context\Element\Column;
use Amasty\ReportBuilderVirtual\Model\Operator\OperatorRepository;

class ContextListBuilder
{
    /**
     * @var Element\Column[]
     */
    private $columns = [];

    /**
     * @var Element\ElementInterface[]
     */
    private $elements = [];

    /**
     * @var ElementFactory
     */
    private $elementFactory;

    /**
     * @var ContextListFactory
     */
    private $contextListFactory;

    /**
     * @var OperatorRepository
     */
    private $operatorRepository;

    public function __construct(
        ElementFactory $elementFactory,
        ContextListFactory $contextListFactory,
        OperatorRepository $operatorRepository
    ) {
        $this->elementFactory = $elementFactory;
        $this->contextListFactory = $contextListFactory;
        $this->operatorRepository = $operatorRepository;
    }

    /**
     * @param Element\Column $column
     */
    public function addColumn(Element\Column $column): void
    {
        $this->columns[$column->getColumnId()] = $column;
    }

    /**
     * @param Element\ElementInterface $element
     */
    public function addElement(Element\ElementInterface $element): void
    {
        $this->elements[] = $element;

        if ($element->getCategory() === Column::CATEGORY) {
            $this->addColumn($element);
        }
    }

    /**
     * @param array $context = \Amasty\ReportBuilderVirtual\Api\Data\ReportVirtualColumnInterface::getContext
     */
    public function build(array $context): ContextList
    {
        foreach ($context as $elementData) {
            $this->addElement($this->buildElement($elementData));
        }

        $list = $this->contextListFactory->create(
            ['context' => $context, 'elements' => $this->elements, 'columns' => $this->columns]
        );

        $this->clear();

        return $list;
    }

    /**
     * @param array $elementData = [
     *     'category' => 'column'|'brackets'|'input', // element type
     *     'type' => 'text'|'decimal'|'varchar'|'int'|'datetime', // element subtype
     *     'column_id' => 'order.entity_id', // column schema id
     *     'id' => 111, // element id
     *     'order' => 0, // element order
     *     'modifier' => null|['abs'], // unary operators ids
     *     'elements' => [$elementData], // array of elements for brackets
     *     'operation' => 'plus'|null // null - without operator
     * ]
     *
     * @return Element\ElementInterface
     * @throws ContextValidationException
     */
    private function buildElement(array $elementData): Element\ElementInterface
    {
        if (isset($elementData[Brackets::ELEMENTS_KEY])) {
            $childElements = [];
            foreach ($elementData[Brackets::ELEMENTS_KEY] as $childElementData) {
                $childElements[] = $this->buildElement($childElementData);
            }
            $elementData[Brackets::ELEMENTS_KEY] = $childElements;
        }

        $element = $this->elementFactory->create($elementData['category'], $elementData);

        $this->validateOperator($element->getOperation());

        $modifiers = $element->getModifier();
        if ($modifiers !== null) {
            foreach ($modifiers as $modifier) {
                $this->validateOperator($modifier);
            }
        }

        if ($element->getCategory() === Column::CATEGORY) {
            $this->addColumn($element);
        }

        return $element;
    }

    /**
     * @param string|null $operator
     *
     * @return void
     * @throws ContextValidationException
     */
    private function validateOperator(?string $operator): void
    {
        if ($operator !== null && !$this->operatorRepository->isExist($operator)) {
            throw new ContextValidationException(
                __('Unregistered virtual column operator detected! Please, check virtual column operators.')
            );
        }
    }

    /**
     * Clear object for next usage
     */
    public function clear(): void
    {
        $this->elements = [];
        $this->columns = [];
    }
}
