<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Report Builder Virtual Columns (System)
*/

declare(strict_types=1);

namespace Amasty\ReportBuilderVirtual\Model\Backend\Report\DataCollector\Columns;

use Amasty\ReportBuilder\Api\ColumnInterface;
use Amasty\ReportBuilder\Api\Data\ReportColumnInterface;
use Amasty\ReportBuilder\Model\Backend\Report\DataCollector\Columns\ColumnDataCollectorInterface;
use Amasty\ReportBuilderVirtual\Api\Data\ReportVirtualColumnInterfaceFactory;
use Amasty\ReportBuilderVirtual\Model\Context\ContextListBuilder;
use Amasty\ReportBuilderVirtual\Ui\DataProvider\Form\Report\Modifier\Columns\VirtualColumnModifier;
use Magento\Framework\Message\ManagerInterface;

class VirtualColumnContext implements ColumnDataCollectorInterface
{
    public const COLUMN_DATA_CONTEXT = 'context';
    public const COLUMN_DATA_TYPE = 'dataType';

    /**
     * @var ReportVirtualColumnInterfaceFactory
     */
    private $columnFactory;

    /**
     * @var ContextColumn
     */
    private $contextColumn;

    /**
     * @var ManagerInterface
     */
    private $messageManager;

    /**
     * @var ContextListBuilder
     */
    private $contextListBuilder;

    public function __construct(
        ReportVirtualColumnInterfaceFactory $columnFactory,
        ContextColumn $contextColumn,
        ManagerInterface $messageManager,
        ContextListBuilder $contextListBuilder
    ) {
        $this->columnFactory = $columnFactory;
        $this->contextColumn = $contextColumn;
        $this->messageManager = $messageManager;
        $this->contextListBuilder = $contextListBuilder;
    }

    public function collectData(ReportColumnInterface $column, array $columnData): void
    {
        if (empty($columnData[self::COLUMN_DATA_CONTEXT])) {
            if ($columnData[ColumnInterface::TYPE] === VirtualColumnModifier::VIRTUAL_TYPE) {
                $this->messageManager->addWarningMessage(
                    __('The custom column cannot be empty and was removed.')
                );

                $column->delete();
            }

            return;
        }
        $extensions = $column->getExtensionAttributes();
        $virtualColumn = $extensions->getVirtualColumn();
        if ($virtualColumn === null) {
            $virtualColumn = $this->columnFactory->create();
        }
        $context = $columnData[self::COLUMN_DATA_CONTEXT];
        $this->contextColumn->prepareColumns($context);

        // validate context
        $this->contextListBuilder->build($context);

        $virtualColumn->setContext($context);
        if (!empty($columnData[self::COLUMN_DATA_TYPE])) {
            $virtualColumn->setDataType($columnData[self::COLUMN_DATA_TYPE]);
        }

        $extensions->setVirtualColumn($virtualColumn);
    }
}
