<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Report Builder Estimation for Magento 2
*/

declare(strict_types=1);

namespace Amasty\ReportBuilderEstimation\Test\Integration\Model\Indexer\Estimation;

use Amasty\ReportBuilder\Model\Indexer\Stock\Table\Column\GetStaticColumns;
use Amasty\ReportBuilder\Model\ResourceModel\Indexer\Stock\IndexResource as StockIndexResource;
use Amasty\ReportBuilderEstimation\Model\ConfigProvider;
use Amasty\ReportBuilderEstimation\Model\Indexer\Estimation\Indexer;
use Amasty\ReportBuilderEstimation\Model\ResourceModel\Indexer\Estimation\IndexResource;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Sql\ExpressionFactory;
use Magento\Framework\ObjectManagerInterface;
use Magento\Framework\Stdlib\DateTime\DateTime;
use Magento\Sales\Api\Data\OrderItemInterface;
use Magento\TestFramework\Helper\Bootstrap;
use PHPUnit\Framework\TestCase;

class IndexerTest extends TestCase
{
    private const SELECT_LIMIT = 50;

    /**
     * @var ObjectManagerInterface
     */
    private $objectManager;

    /**
     * @var ResourceConnection
     */
    private $resourceConnection;

    protected function setUp(): void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->resourceConnection = $this->objectManager->get(ResourceConnection::class);
    }

    /**
     * @magentoDbIsolation disabled
     * @magentoAppArea global
     * @magentoDataFixture Magento/Sales/_files/order_configurable_product_rollback.php
     * @magentoDataFixture Magento/Sales/_files/order_configurable_product.php
     * @magentoDataFixture Amasty_ReportBuilderEstimation::Test/Integration/_files/stock_data_rollback.php
     * @magentoDataFixture Amasty_ReportBuilderEstimation::Test/Integration/_files/stock_data.php
     * @magentoConfigFixture default_store amasty_report_builder/general/estimation_days 3
     */
    public function testExecute(): void
    {
        /** @var Indexer $indexer */
        $indexer = $this->objectManager->get(Indexer::class);
        $indexer->execute();

        /** @var ConfigProvider $configProvider */
        $configProvider = $this->objectManager->get(ConfigProvider::class);
        $estimationDays = $configProvider->getEstimationDays();

        $select = $this->resourceConnection->getConnection()->select()
            ->from($this->resourceConnection->getTableName(IndexResource::MAIN_TABLE))
            ->limit(self::SELECT_LIMIT);
        $estimationData = $this->resourceConnection->getConnection()->fetchAll($select);

        $productIds = array_column($estimationData, IndexResource::PRODUCT_ID_COLUMN);

        $orderItemData = $this->getOrderItemData($productIds, $estimationDays);
        $stockData = $this->getStockData($productIds);

        foreach ($estimationData as $row) {
            $stockQty = $stockData[$row[IndexResource::PRODUCT_ID_COLUMN]];
            $orderedQty = $orderItemData[$row[IndexResource::PRODUCT_ID_COLUMN]];

            $expectedAvgSales = $orderedQty / $estimationDays;
            $this->assertEquals(
                round($expectedAvgSales, 4),
                $row[IndexResource::AVG_SALES_COLUMN]
            );

            $expectedStockThreshold = $stockQty * $estimationDays / $expectedAvgSales;
            $this->assertEquals(
                round($expectedStockThreshold, 4),
                $row[IndexResource::STOCK_THRESHOLD_COLUMN]
            );
        }
    }

    protected function tearDown(): void
    {
        $this->resourceConnection->getConnection()->truncateTable(
            $this->resourceConnection->getTableName(IndexResource::MAIN_TABLE)
        );
        $this->resourceConnection->getConnection()->truncateTable(
            $this->resourceConnection->getTableName(IndexResource::REPLICA_TABLE)
        );
    }

    /**
     * @param int[] $productIds
     * @param int $estimationDays
     * @return array
     */
    private function getOrderItemData(array $productIds, int $estimationDays): array
    {
        /** @var DateTime $dateTime */
        $dateTime = $this->objectManager->get(DateTime::class);
        $fromDate = $dateTime->gmtDate(null, sprintf('-%d days', $estimationDays));

        /** @var ExpressionFactory $expressionFactory */
        $expressionFactory = $this->objectManager->get(ExpressionFactory::class);

        $orderItemDataSelect = $this->resourceConnection->getConnection()->select()->from(
            $this->resourceConnection->getTableName('sales_order_item'),
            [
                OrderItemInterface::PRODUCT_ID,
                $expressionFactory->create([
                    'expression' => sprintf('SUM(%s)', OrderItemInterface::QTY_ORDERED)
                ])
            ]
        )->where(
            OrderItemInterface::PRODUCT_ID . ' IN (?)',
            $productIds
        )->where(
            OrderItemInterface::CREATED_AT . '>= ?',
            $fromDate
        )->group(OrderItemInterface::PRODUCT_ID);

        return $this->resourceConnection->getConnection()->fetchPairs($orderItemDataSelect);
    }

    /**
     * @param int[] $productIds
     * @return array
     */
    private function getStockData(array $productIds): array
    {
        $stockDataSelect = $this->resourceConnection->getConnection()->select()->from(
            $this->resourceConnection->getTableName(StockIndexResource::MAIN_TABLE),
            [GetStaticColumns::PRODUCT_ID_COLUMN, GetStaticColumns::TOTAL_QTY_COLUMN]
        )->where(
            GetStaticColumns::PRODUCT_ID_COLUMN . ' IN (?)',
            $productIds
        );

        return $this->resourceConnection->getConnection()->fetchPairs($stockDataSelect);
    }
}
