<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Report Builder Estimation for Magento 2
*/

declare(strict_types=1);

namespace Amasty\ReportBuilderEstimation\Model\ResourceModel\Indexer\Estimation;

use Exception;
use Magento\Catalog\Model\ResourceModel\Indexer\ActiveTableSwitcher;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\DB\Select;
use Magento\Framework\Indexer\Table\Strategy as IndexerTableStrategy;

class TableWorker
{
    /**
     * @var IndexResource
     */
    private $indexResource;

    /**
     * @var IndexerTableStrategy
     */
    private $indexerTableStrategy;

    /**
     * @var ActiveTableSwitcher
     */
    private $activeTableSwitcher;

    public function __construct(
        IndexResource $indexResource,
        IndexerTableStrategy $indexerTableStrategy,
        ActiveTableSwitcher $activeTableSwitcher
    ) {
        $this->indexResource = $indexResource;
        $this->indexerTableStrategy = $indexerTableStrategy;
        $this->activeTableSwitcher = $activeTableSwitcher;
    }

    private function getConnection(): AdapterInterface
    {
        return $this->indexResource->getConnection();
    }

    public function getIdxTable(): string
    {
        $this->indexerTableStrategy->setUseIdxTable(true);
        return $this->indexerTableStrategy->prepareTableName(IndexResource::REPLICA_TABLE);
    }

    public function createTemporaryTable(): void
    {
        $this->getConnection()->createTemporaryTableLike(
            $this->indexResource->getTableName($this->getIdxTable()),
            $this->indexResource->getTableName(IndexResource::REPLICA_TABLE),
            true
        );
    }

    /**
     * Populate temp table from select.
     */
    public function insertToTemporaryTable(Select $select): void
    {
        $query = $select->insertFromSelect(
            $this->indexResource->getTableName($this->getIdxTable()),
            [IndexResource::PRODUCT_ID_COLUMN, IndexResource::AVG_SALES_COLUMN, IndexResource::STOCK_THRESHOLD_COLUMN]
        );
        $this->getConnection()->query($query);
    }

    /**
     * Move data from temp table to replica table.
     *
     * @throws Exception
     */
    public function syncDataFull(): void
    {
        $this->syncData($this->indexResource->getTableName(IndexResource::REPLICA_TABLE));
    }

    /**
     * Move data from temp table to passed table.
     *
     * @param string $destinationTable
     * @throws Exception
     */
    private function syncData(string $destinationTable): void
    {
        $connection = $this->getConnection();

        $connection->beginTransaction();
        try {
            $connection->delete($destinationTable);
            $this->insertFromTable(
                $this->indexResource->getTableName($this->getIdxTable()),
                $destinationTable
            );
            $connection->commit();
        } catch (Exception $e) {
            $connection->rollBack();
            throw $e;
        }
    }

    /**
     * Switch replica table with main table.
     *
     * @return void
     */
    public function switchTables(): void
    {
        $this->activeTableSwitcher->switchTable(
            $this->getConnection(),
            [$this->indexResource->getTableName(IndexResource::MAIN_TABLE)]
        );
    }

    private function insertFromTable(string $sourceTable, string $destTable): void
    {
        $sourceColumns = array_keys($this->getConnection()->describeTable($sourceTable));
        $targetColumns = array_keys($this->getConnection()->describeTable($destTable));

        $select = $this->getConnection()->select()->from($sourceTable, $sourceColumns);

        $this->getConnection()->query($this->getConnection()->insertFromSelect($select, $destTable, $targetColumns));
    }
}
