<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Report Builder Estimation for Magento 2
*/

declare(strict_types=1);

namespace Amasty\ReportBuilderEstimation\Model\ResourceModel\Indexer\Estimation;

use Amasty\ReportBuilder\Model\Indexer\Stock\Table\Column\GetStaticColumns;
use Amasty\ReportBuilder\Model\ResourceModel\Indexer\Stock\IndexResource as StockIndexResource;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Select;
use Magento\Framework\DB\Sql\Expression;
use Magento\Framework\DB\Sql\ExpressionFactory;
use Magento\Framework\Stdlib\DateTime\DateTime;

/**
 * Use columns from StockIndexResource.
 * @see StockIndexResource
 *
 * Depends on Amasty Report Builder Stock Indexer.
 * Generate select for amasty_report_builder_sales_estimation table.
 */
class GetIndexSelect
{
    /**
     * @var ResourceConnection
     */
    private $resourceConnection;

    /**
     * @var DateTime
     */
    private $dateTime;

    /**
     * @var ExpressionFactory
     */
    private $expressionFactory;

    public function __construct(
        ResourceConnection $resourceConnection,
        DateTime $dateTime,
        ExpressionFactory $expressionFactory
    ) {
        $this->resourceConnection = $resourceConnection;
        $this->dateTime = $dateTime;
        $this->expressionFactory = $expressionFactory;
    }

    public function execute(int $estimationDays): Select
    {
        $connection = $this->resourceConnection->getConnection();

        $fromDate = $this->dateTime->gmtDate(null, sprintf('-%d days', $estimationDays));

        $avgSalesExpression = $this->expressionFactory->create([
            'expression' => sprintf('ROUND(SUM(qty_ordered)/%d, 4)', $estimationDays)
        ]);

        return $connection->select()->from(
            ['cpe' => $this->resourceConnection->getTableName('catalog_product_entity')],
            [
                IndexResource::PRODUCT_ID_COLUMN => 'entity_id',
                IndexResource::AVG_SALES_COLUMN => $avgSalesExpression,
                IndexResource::STOCK_THRESHOLD_COLUMN => $this->getStockThresholdExpression(
                    $estimationDays,
                    $avgSalesExpression
                )
            ]
        )->joinLeft(
            ['soi' => $this->resourceConnection->getTableName('sales_order_item')],
            'cpe.entity_id = soi.product_id',
            []
        )->join(
            ['stock' => $this->resourceConnection->getTableName(StockIndexResource::MAIN_TABLE)],
            sprintf('soi.product_id = stock.%s', GetStaticColumns::PRODUCT_ID_COLUMN),
            []
        )->where(
            'soi.created_at >= ?',
            $fromDate
        )->group('cpe.entity_id');
    }

    /**
     * Generate expression for stock_threshold column.
     * Based on total qty, estimation days and average sales.
     */
    private function getStockThresholdExpression(int $estimationDays, Expression $avgSalesExpression): Expression
    {
        return $this->expressionFactory->create([
            'expression' => sprintf(
                'FLOOR(%s * %d / (%s))',
                sprintf(
                    '%s.%s',
                    'stock',
                    GetStaticColumns::TOTAL_QTY_COLUMN
                ),
                $estimationDays,
                $avgSalesExpression
            )
        ]);
    }
}
