define([
    'underscore',
    'amcharts'
], function (_, amcharts) {
    'use strict';

    var model = {
        options: {
            xAxisType: ''
        },

        /**
         * Axis Series initialization
         *
         * @private
         * @returns {void}
         */
        _initSeries: function () {
            if (this.chart.series.length) {
                this.chart.series.removeIndex(0);
            }

            this.series = new window.am4charts.ColumnSeries();
        },

        /**
         * Axis initialization
         *
         * @private
         * @param {String} axis
         * @param {String} type
         * @returns {void}
         */
        _initAxis: function (axis, type) {
            if (this.chart[axis + 'Axes'].length) {
                this.chart[axis + 'Axes'].removeIndex(0);
            }

            switch (type) {
                case 'smallint':
                case 'decimal':
                    this.initTypeValue(axis);
                    break;
                case 'text':
                case 'varchar':
                case 'int':
                    this.initTypeCategory(axis);
                    break;
                default:
                    this.initTypeDate(axis);
            }
        },

        /**
         * Chart rendering
         *
         * @private
         * @param {Object} response
         * @returns {void}
         */
        _renderChart: function (response) {
            this.chart.xAxes.push(this.valueAxisX);
            this.chart.yAxes.push(this.valueAxisY);
            this.chart.series.push(this.series);

            model._applyConfig.call(this, response);
        },

        /**
         * @private
         * @returns {void}
         */
        _getHorizontalConfig: function () {
            var labelBullet = this.series.bullets.push(new amcharts.LabelBullet());

            labelBullet.label.horizontalCenter = 'right';
            labelBullet.label.dx = -10;
            labelBullet.label.text = '{valueX}';

            this.series.columns.template.column.cornerRadiusTopRight = 5;
            this.series.columns.template.column.cornerRadiusBottomRight = 5;
            this.valueAxisX.renderer.grid.template.location = 0;
            this.valueAxisX.renderer.grid.template.disabled = true;
            this.valueAxisY.renderer.minGridDistance = 1;
            this.valueAxisY.renderer.inversed = true;
        },

        /**
         * @private
         * @returns {void}
         */
        _getVerticalConfig: function () {
            var labelBullet = this.series.bullets.push(new amcharts.LabelBullet());

            labelBullet.label.dy = 20;
            labelBullet.label.text = '{valueY}';

            this.series.columns.template.column.cornerRadiusTopRight = 5;
            this.series.columns.template.column.cornerRadiusTopLeft = 5;
            this.valueAxisX.renderer.minGridDistance = 60;
            this.valueAxisX.renderer.labels.template.maxWidth = 150;
            this.valueAxisX.renderer.labels.template.tooltipText = '{valueX}';
        },

        /**
         * Apply chart config from xml and set additional settings based on chart type
         *
         * @private
         * @param {Object} response
         * @returns {void}
         */
        _applyConfig: function (response) {
            var config = this.chart_types[this.chartType];

            if (_.isUndefined(config)) {
                return;
            }

            this.setChartConfig(config.settings);

            this.valueAxisX.renderer.labels.template.wrap = true;
            this.valueAxisX.renderer.labels.template.truncate = true;

            this.valueAxisY.renderer.labels.template.wrap = true;
            this.valueAxisY.renderer.labels.template.maxWidth = 150;

            switch (model.options.xAxisType) {
                case 'smallint':
                case 'decimal':
                    model._getHorizontalConfig.call(this);
                    this.sortBySeries(this.valueAxisY);
                    break;
                case 'text':
                case 'varchar':
                case 'int':
                default:
                    model._getVerticalConfig.call(this);
                    this.sortBySeries(this.valueAxisX);
                    break;
            }

            this.series.columns.template.adapter.add('fill', function (fill, target) {
                return this.chart.colors.getIndex(target.dataItem.index);
            }.bind(this));

            this.setLabelsOffsetX(this.valueAxisX, response.data);
        }
    };

    return {
        /**
         * @public
         * @param {Object} node
         * @returns {void}
         */
        init: function (node) {
            this.chart = window.am4core.create(node, amcharts.XYChart);

            this.initScrollbar();
        },

        /**
         * @public
         * @param {Object} response
         * @returns {void}
         */
        update: function (response) {
            model.options.xAxisType = response.xAxisType;

            model._initSeries.call(this);
            model._initAxis.call(this, 'x', response.xAxisType);
            model._initAxis.call(this, 'y', response.yAxisType);
            model._renderChart.call(this, response);
        }
    };
});
