<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2: Stripe (System)
 */

namespace Amasty\RecurringStripe\Plugin\Gateway\Http\Client;

use Amasty\RecurringStripe\Model\StripeAdapterProvider;
use Amasty\RecurringStripe\Plugin\Gateway\Helper\AmountHelper;
use Amasty\Stripe\Gateway\Http\Client\AbstractClient;
use Amasty\Stripe\Gateway\Http\TransferFactory;
use Magento\Payment\Gateway\Http\TransferInterface;

class AbstractClientPlugin
{
    /**
     * @var TransferFactory
     */
    private $transferFactory;

    /**
     * @var StripeAdapterProvider
     */
    private $stripeAdapterProvider;

    public function __construct(TransferFactory $transferFactory, StripeAdapterProvider $stripeAdapterProvider)
    {
        $this->transferFactory = $transferFactory;
        $this->stripeAdapterProvider = $stripeAdapterProvider;
    }

    /**
     * @param AbstractClient $subject
     * @param \Closure $proceed
     * @param TransferInterface $transferObject
     *
     * @return array
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function aroundPlaceRequest(
        AbstractClient $subject,
        \Closure $proceed,
        TransferInterface $transferObject
    ) {
        $needRefund = false;
        $needCancel = false;

        if ($body = $transferObject->getBody()) {
            if (!isset($body['chargeId'])
                && isset($body['amount'])
                && $body['amount'] == AmountHelper::STRIPE_MIN_AMOUNT
            ) {
                $body['description'] = __('This is a test payment for authorization purposes to verify the validity '
                    . 'of the card. Refund is issued immediately but its processing may take some time.');
                $transferObject = $this->transferFactory->create($body);
                if ($body['capture']) {
                    $needRefund = true;
                } else {
                    $needCancel = true;
                }
            }
        }
        $result = $proceed($transferObject);

        if ($needRefund && isset($result['object'])) {
            $charge = $this->getCharge($result['object']);
            $this->createRefund($charge);
        }

        if ($needCancel && isset($result['object'])) {
            $result['object']->cancel();
        }

        return $result;
    }

    /**
     * @param \Stripe\PaymentIntent $paymentIntent
     *
     * @return \Stripe\Collection
     */
    private function getCharge(\Stripe\PaymentIntent $paymentIntent)
    {
        return \Stripe\Charge::all(
            [
                'payment_intent' => $paymentIntent->id,
                'limit' => 1,
            ]
        );
    }

    /**
     * @param \Stripe\Collection $charge
     */
    private function createRefund($charge): void
    {
        $stripeAdapter = $this->stripeAdapterProvider->get();
        $stripeAdapter->refundCreate(['charge' => $charge->data[0]->id]);
    }
}
