<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2: Stripe (System)
 */

namespace Amasty\RecurringStripe\Model\Subscription;

use Amasty\RecurringPayments\Api\Subscription\SubscriptionInfoInterface;
use Amasty\RecurringPayments\Api\Subscription\SubscriptionInfoInterfaceFactory;
use Amasty\RecurringPayments\Api\TransactionRepositoryInterface;
use Amasty\RecurringPayments\Model\DateTime\DateTimeComparer;
use Amasty\RecurringPayments\Model\Repository\AddressRepository;
use Amasty\RecurringPayments\Model\Subscription\GridSource;
use Amasty\RecurringPayments\Model\Subscription\Scheduler\DateTimeInterval;
use Amasty\RecurringStripe\Api\ProductRepositoryInterface as StripeProductRepository;
use Amasty\RecurringStripe\Api\CustomerRepositoryInterface;
use Amasty\RecurringPayments\Api\Subscription\RepositoryInterface;
use Amasty\RecurringPayments\Api\Subscription\SubscriptionInterface;
use Amasty\RecurringPayments\Api\Subscription\SubscriptionInterfaceFactory;
use Amasty\RecurringPayments\Api\Subscription\GridInterface;
use Amasty\RecurringPayments\Model\Date;
use Amasty\RecurringStripe\Model\Subscription\Cache as SubscriptionCache;
use Amasty\RecurringStripe\Model\StripeAdapterProvider;
use Amasty\Stripe\Api\Data\CustomerInterface;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Directory\Model\CountryFactory;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Pricing\PriceCurrencyInterface;
use Magento\Framework\UrlInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\OrderFactory;

// @TODO: refactor that with all payment addons
class Grid extends GridSource implements GridInterface
{
    public const ACTIVE_STATUSES = [
        StatusMapper::ACTIVE,
        StatusMapper::TRIAL,
    ];
    public const MAX_LIMIT = 100;

    /**
     * @var array
     */
    private $stripeProducts = [];

    /**
     * @var SubscriptionInterfaceFactory
     */
    private $subscriptionFactory;

    /**
     * @var StripeAdapterProvider
     */
    private $adapterProvider;

    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;

    /**
     * @var StatusMapper
     */
    private $statusMapper;

    /**
     * @var Date
     */
    private $date;

    /**
     * @var OrderFactory
     */
    private $orderFactory;

    /**
     * @var UrlInterface
     */
    private $urlBuilder;

    /**
     * @var RepositoryInterface
     */
    private $subscriptionRepository;

    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * @var StripeProductRepository
     */
    private $stripeProductRepository;

    /**
     * @var AddressRepository
     */
    private $addressRepository;

    /**
     * @var SubscriptionInfoInterfaceFactory
     */
    private $subscriptionInfoFactory;

    /**
     * @var Cache
     */
    private $subscriptionCache;

    /**
     * @var InvoiceInfoFactory
     */
    private $infoFactory;

    /**
     * @var TransactionRepositoryInterface
     */
    private $transactionRepository;

    /**
     * @var DateTimeInterval
     */
    private $dateTimeInterval;

    /**
     * @var DateTimeComparer
     */
    private $dateTimeComparer;

    public function __construct(
        Date $date,
        PriceCurrencyInterface $priceCurrency,
        CountryFactory $countryFactory,
        SubscriptionInterfaceFactory $subscriptionFactory,
        StripeAdapterProvider $adapterProvider,
        CustomerRepositoryInterface $customerRepository,
        StatusMapper $statusMapper,
        OrderFactory $orderFactory,
        UrlInterface $urlBuilder,
        RepositoryInterface $subscriptionRepository,
        ProductRepositoryInterface $productRepository,
        StripeProductRepository $stripeProductRepository,
        AddressRepository $addressRepository,
        SubscriptionInfoInterfaceFactory $subscriptionInfoFactory,
        InvoiceInfoFactory $infoFactory,
        SubscriptionCache $subscriptionCache,
        TransactionRepositoryInterface $transactionRepository,
        DateTimeInterval $dateTimeInterval,
        DateTimeComparer $dateTimeComparer
    ) {
        parent::__construct($date, $priceCurrency, $countryFactory);
        $this->subscriptionFactory = $subscriptionFactory;
        $this->adapterProvider = $adapterProvider;
        $this->customerRepository = $customerRepository;
        $this->statusMapper = $statusMapper;
        $this->orderFactory = $orderFactory;
        $this->urlBuilder = $urlBuilder;
        $this->subscriptionRepository = $subscriptionRepository;
        $this->productRepository = $productRepository;
        $this->stripeProductRepository = $stripeProductRepository;
        $this->addressRepository = $addressRepository;
        $this->subscriptionInfoFactory = $subscriptionInfoFactory;
        $this->subscriptionCache = $subscriptionCache;
        $this->infoFactory = $infoFactory;
        $this->transactionRepository = $transactionRepository;
        $this->dateTimeInterval = $dateTimeInterval;
        $this->dateTimeComparer = $dateTimeComparer;
    }

    /**
     * @param int $customerId
     * @return SubscriptionInterface[]
     */
    public function process(int $customerId)
    {
        $subscriptions = [];

        /** @var CustomerInterface $customer */
        try {
            $adapter = $this->adapterProvider->get();
            $customer = $this->customerRepository->getStripeCustomer($customerId, $adapter->getAccountId());
        } catch (NoSuchEntityException $e) {
            return $subscriptions;
        }

        /** @var \Stripe\Collection $subscriptionsStripe */
        $subscriptionsStripe = $adapter->subscriptionList([
            'customer'          => $customer->getStripeCustomerId(),
            'collection_method' => 'charge_automatically',
            'status'            => 'all',
            'limit'             => self::MAX_LIMIT
        ]);

        if (empty($subscriptionsStripe->data)) {
            return $subscriptions;
        }

        $subscriptionIds = [];
        /** @var \Stripe\Subscription $subscriptionStripe */
        foreach ($subscriptionsStripe->data as $subscriptionStripe) {
            $subscriptionIds[] = $subscriptionStripe->id;
        }

        $lastTransactions = $this->transactionRepository->getLastRelatedTransactions($subscriptionIds);

        /** @var \Stripe\Subscription $subscriptionStripe */
        foreach ($subscriptionsStripe->data as $subscriptionStripe) {
            /** @var SubscriptionInfoInterface $subscriptionInfo */
            $subscriptionInfo = $this->subscriptionInfoFactory->create();

            try {
                /** @var SubscriptionInterface $subscription */
                $subscription = $this->subscriptionRepository->getBySubscriptionId($subscriptionStripe->id);
            } catch (NoSuchEntityException $exception) {
                /** @var SubscriptionInterface $subscription */
                $subscription = $this->subscriptionFactory->create();
                $subscription->setSubscriptionId($subscriptionStripe->id);
                $subscription->setStartDate(\date('Y-m-d H:i:s', $subscriptionStripe->created));
            }

            if ($subscription->getId()) {
                $lastTransaction = $lastTransactions[$subscription->getSubscriptionId()] ?? null;
                if ($lastTransaction && $subscription->getLastPaymentDate()) {
                    $subscriptionInfo->setLastBilling(
                        $this->formatDate(strtotime($subscription->getLastPaymentDate()))
                    );
                    $subscriptionInfo->setLastBillingAmount(
                        $this->formatPrice(
                            (float)$lastTransaction->getBillingAmount(),
                            $lastTransaction->getBillingCurrencyCode()
                        )
                    );
                }
            } elseif ($subscriptionStripe->latest_invoice) {
                // backward compatibility,if we didn't save sub to our table
                $latestInvoice = $this->getInvoiceInfoFromStripe((string)$subscriptionStripe->latest_invoice);
                if ($latestInvoice->getAmount() > 0.0001) {
                    $subscriptionInfo->setLastBilling($this->formatDate((int)$latestInvoice->getDate()));
                    $subscriptionInfo->setLastBillingAmount($this->getBillingAmount($latestInvoice));
                }
            }

            $isTrialFake = !$subscription->getTrialDays();

            $subscriptionInfo->setSubscription($subscription);
            if ($address = $this->findAddress($subscription, $subscriptionStripe->id)) {
                $this->setStreet($address);
                $this->setCountry($address);
                $subscriptionInfo->setAddress($address);
            }

            if (in_array($subscriptionStripe->status, self::ACTIVE_STATUSES)) {
                $this->setNextInvoice($subscription, $subscriptionInfo);
                $subscriptionInfo->setIsActive(true);
            } else {
                $subscriptionInfo->setIsActive(false);
            }

            $subscriptionInfo->setOrderIncrementId((string)$subscriptionStripe->metadata->increment_id);
            $subscriptionInfo->setOrderLink($this->getOrderLink((string)$subscriptionStripe->metadata->increment_id));
            $subscriptionInfo->setSubscriptionName($this->getProductName($subscriptionStripe->plan));
            $subscriptionInfo->setStartDate($this->formatDate(strtotime($subscription->getStartDate())));

            $subscription->setQty($subscriptionStripe->quantity);
            $subscription->setDelivery((string)$subscriptionStripe->metadata->delivery);
            !$isTrialFake && $this->setTrial($subscriptionInfo, $subscriptionStripe);

            $status = $subscriptionStripe->status;
            if ($isTrialFake && $status == StatusMapper::TRIAL) {
                $status = StatusMapper::ACTIVE;
            }
            $subscriptionInfo->setStatus($this->statusMapper->getStatus($status));
            $subscription->setPaymentMethod(\Amasty\Stripe\Model\Ui\ConfigProvider::CODE);
            $subscriptions[] = $subscriptionInfo;
        }

        return $subscriptions;
    }

    /**
     * @param SubscriptionInterface $subscription
     * @param string $subscriptionId
     * @return \Amasty\RecurringPayments\Api\Subscription\AddressInterface|null
     */
    private function findAddress(SubscriptionInterface $subscription, string $subscriptionId)
    {
        if ($addressId = $subscription->getAddressId()) { // Since 1.1.1
            try {
                return $this->addressRepository->getById($addressId);
            } catch (NoSuchEntityException $exception) {
                return null;
            }
        } else { // Compatibility with pre 1.1.1
            try {
                return $this->addressRepository->getBySubscriptionId($subscriptionId);
            } catch (NoSuchEntityException $exception) {
                return null;
            }
        }
    }

    /**
     * @param string $invoiceId
     * @return InvoiceInfo
     */
    private function getInvoiceInfoFromStripe(string $invoiceId): InvoiceInfo
    {
        $info = $this->subscriptionCache->getInvoiceInfo($invoiceId);

        if (!$info) {
            $adapter = $this->adapterProvider->get();
            /** @var \Stripe\Invoice $invoice */
            $invoice = $adapter->invoiceRetrieve($invoiceId);
            /** @var InvoiceInfo $info */
            $info = $this->infoFactory->create();
            $info->setId((string)$invoiceId)
                ->setDate((int)$invoice->date)
                ->setAmount((float)$invoice->amount_due)
                ->setCurrency((string)$invoice->currency);
            $this->subscriptionCache->saveInvoiceInfo($invoiceId, $info);
        }

        return $info;
    }

    /**
     * @param string $subscriptionId
     * @return InvoiceInfo
     */
    private function getUpcomingInvoiceInfo(string $subscriptionId): InvoiceInfo
    {
        $info = $this->subscriptionCache->getInvoiceInfo($subscriptionId);

        if (!$info || $info->getDate() < time()) { // Invalidate cache if next billing date is in past
            $adapter = $this->adapterProvider->get();
            /** @var \Stripe\Invoice $invoice */
            $invoice = $adapter->upcomingInvoiceRetrieve(['subscription' => $subscriptionId]);

            /** @var InvoiceInfo $info */
            $info = $this->infoFactory->create();
            $info->setId('')
                ->setDate((int)$invoice->date)
                ->setAmount((float)$invoice->amount_due)
                ->setCurrency((string)$invoice->currency);
            $this->subscriptionCache->saveInvoiceInfo($subscriptionId, $info);
        }

        return $info;
    }

    /**
     * @param SubscriptionInfoInterface $subscriptionInfo
     * @param \Stripe\Subscription $subscriptionStripe
     */
    private function setTrial(SubscriptionInfoInterface $subscriptionInfo, \Stripe\Subscription $subscriptionStripe)
    {
        if ($subscriptionStripe->status === StatusMapper::TRIAL) {
            $subscription = $subscriptionInfo->getSubscription();
            $startDateTimestamp = strtotime($subscription->getStartDate());
            if ($startDateTimestamp > $subscriptionStripe->trial_start) {
                $subscriptionInfo->setTrialStartDate($this->formatDate($startDateTimestamp));
            } else {
                $subscriptionInfo->setTrialStartDate($this->formatDate($subscriptionStripe->trial_start));
            }
            $subscriptionInfo->setTrialEndDate($this->formatDate($subscriptionStripe->trial_end));
        }
    }

    /**
     * @param string $incrementId
     * @return string
     */
    private function getOrderLink(string $incrementId): string
    {
        /** @var Order $order */
        $order = $this->orderFactory->create();

        $order->loadByIncrementId($incrementId);

        return $this->urlBuilder->getUrl('sales/order/view', ['order_id' => $order->getId()]);
    }

    /**
     * @param SubscriptionInterface $subscription
     * @param SubscriptionInfoInterface $subscriptionInfo
     * @throws \Exception
     */
    private function setNextInvoice(
        SubscriptionInterface $subscription,
        SubscriptionInfoInterface $subscriptionInfo
    ): void {
        try {
            $invoiceInfo = $this->getUpcomingInvoiceInfo($subscriptionInfo->getSubscription()->getSubscriptionId());
        } catch (\Exception $e) {
            return;
        }

        if ($subscription->getId()) {
            $lastPaymentDate = $subscription->getLastPaymentDate();
            if ($lastPaymentDate) {
                $nextBillingDate = $this->dateTimeInterval->getNextBillingDate(
                    $lastPaymentDate,
                    $subscription->getFrequency(),
                    $subscription->getFrequencyUnit()
                );
            } elseif ($subscription->getTrialDays()) {
                $nextBillingDate = $this->dateTimeInterval->getStartDateAfterTrial(
                    $subscription->getStartDate(),
                    $subscription->getTrialDays()
                );
            } elseif (!$this->dateTimeComparer->compareDates(
                $subscription->getCreatedAt(),
                $subscription->getStartDate()
            )) {
                $nextBillingDate = $subscription->getStartDate();
            } else {
                $nextBillingDate = $this->dateTimeInterval->getNextBillingDate(
                    $subscription->getStartDate(),
                    $subscription->getFrequency(),
                    $subscription->getFrequencyUnit()
                );
            }

            $subscriptionEndDate = $subscription->getEndDate();
            $isNextDateExists = true;
            if ($subscriptionEndDate) {
                $subscriptionEndDateObject = new \DateTime($subscriptionEndDate);
                $nextBillingDateObject = new \DateTime($nextBillingDate);
                if ($nextBillingDateObject > $subscriptionEndDateObject) {
                    $isNextDateExists = false;
                }
            }

            if ($isNextDateExists) {
                $subscriptionInfo->setNextBilling($this->formatDate(strtotime($nextBillingDate)));
                $baseNextBillingAmount = (float)$subscription->getBaseGrandTotalWithDiscount();

                if ($subscription->getRemainingDiscountCycles() !== null
                    && $subscription->getRemainingDiscountCycles() < 1
                ) {
                    $baseNextBillingAmount = (float)$subscription->getBaseGrandTotal();
                }

                $subscriptionInfo->setNextBillingAmount(
                    $this->formatPrice($baseNextBillingAmount, mb_strtoupper($invoiceInfo->getCurrency()))
                );
            }

            return;
        }

        $subscriptionInfo->setNextBilling($this->formatDate((int)$invoiceInfo->getDate()));
        $subscriptionInfo->setNextBillingAmount($this->getBillingAmount($invoiceInfo));
    }

    /**
     * @param \Stripe\Plan $plan
     * @return string
     */
    private function getProductName(\Stripe\Plan $plan): string
    {
        $productId = $plan->product;

        if (!isset($this->stripeProducts[$productId])) {
            $stripeProduct = $this->stripeProductRepository->getByStripeProdId($productId);
            /** @var \Magento\Catalog\Api\Data\ProductInterface $product */
            $product = $this->productRepository->getById($stripeProduct->getProductId());
            $this->stripeProducts[$productId] = $product->getName();
        }

        return $this->stripeProducts[$productId];
    }

    /**
     * @param InvoiceInfo $invoiceInfo
     * @return string
     */
    private function getBillingAmount(InvoiceInfo $invoiceInfo): string
    {
        return $this->formatPrice(
            $invoiceInfo->getAmount() / \Amasty\RecurringPayments\Model\Amount::PERCENT,
            mb_strtoupper($invoiceInfo->getCurrency())
        );
    }
}
