<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2: Stripe (System)
 */

namespace Amasty\RecurringStripe\Model\Subscription\Create;

use Amasty\RecurringPayments\Api\Subscription\SubscriptionInterface;
use Amasty\RecurringPayments\Model\Subscription\Create\CreateSubscriptionHandlerInterface;
use Amasty\RecurringStripe\Api\ProductRepositoryInterface;
use Amasty\RecurringStripe\Model\Adapter;
use Amasty\RecurringStripe\Model\Processors\CreatePlan;
use Amasty\RecurringStripe\Model\Processors\CreateProduct;
use Amasty\RecurringStripe\Model\Processors\CreateSubscription;
use Amasty\RecurringStripe\Model\StripeProduct;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Model\Quote\Item\AbstractItem;
use Magento\Sales\Api\Data\OrderInterface;

class CreateSubscriptionHandler implements CreateSubscriptionHandlerInterface
{
    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * @var CreateProduct
     */
    private $createProduct;

    /**
     * @var CreatePlan
     */
    private $createPlan;

    /**
     * @var CreateSubscription
     */
    private $createSubscription;

    /**
     * @var Adapter
     */
    private $adapter;

    public function __construct(
        ProductRepositoryInterface $productRepository,
        CreateProduct $createProduct,
        CreatePlan $createPlan,
        CreateSubscription $createSubscription,
        Adapter $adapter
    ) {
        $this->productRepository = $productRepository;
        $this->createProduct = $createProduct;
        $this->createPlan = $createPlan;
        $this->createSubscription = $createSubscription;
        $this->adapter = $adapter;
    }

    /**
     * @param OrderInterface $order
     * @param AbstractItem $item
     * @param SubscriptionInterface $subscription
     */
    public function handle(OrderInterface $order, AbstractItem $item, SubscriptionInterface $subscription): void
    {
        $productId = $item->getProduct()->getId();

        try {
            /** @var StripeProduct $product */
            $product = $this->productRepository->getByProductId($productId, $this->adapter->getAccountId());
        } catch (NoSuchEntityException $exception) {
            $product = $this->createProduct->execute($item, (int)$productId);
        }

        /** @var \Stripe\Plan $plan */
        $plan = $this->createPlan->execute($subscription, $item, (string)$product->getStripeProductId());
        $subscriptionId = $this->createSubscription->execute($subscription, $item, $order, $plan);
        $subscription->setSubscriptionId($subscriptionId);
    }
}
