<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2: Stripe (System)
 */

namespace Amasty\RecurringStripe\Model\IpnHandlers\Invoice;

use Amasty\RecurringPayments\Api\Generators\RecurringTransactionGeneratorInterface;
use Amasty\RecurringPayments\Api\Subscription\RepositoryInterface;
use Amasty\RecurringPayments\Api\Subscription\RepositoryInterface as SubscriptionRepositoryInterface;
use Amasty\RecurringPayments\Model\Config;
use Amasty\RecurringPayments\Model\Config\Source\Status;
use Amasty\RecurringPayments\Model\Date;
use Amasty\RecurringPayments\Model\Subscription\Email\EmailNotifier;
use Amasty\RecurringPayments\Model\Subscription\HandleOrder\CompositeHandler;
use Amasty\RecurringPayments\Model\Subscription\HandleOrder\CompositeHandlerFactory;
use Amasty\RecurringPayments\Model\Subscription\HandleOrder\HandleOrderContext;
use Amasty\RecurringPayments\Model\Subscription\HandleOrder\HandleOrderContextFactory;
use Amasty\RecurringStripe\Model\StripeAdapterProvider;
use Magento\Framework\Stdlib\DateTime\DateTime;

class PaymentSucceeded extends AbstractInvoice
{
    /**
     * @var StripeAdapterProvider
     */
    private $adapterProvider;

    /**
     * @var SubscriptionRepositoryInterface
     */
    private $subscriptionRepository;

    /**
     * @var CompositeHandlerFactory
     */
    private $compositeHandlerFactory;

    /**
     * @var HandleOrderContextFactory
     */
    private $handleOrderContextFactory;

    /**
     * @var DateTime
     */
    private $dateTime;

    public function __construct(
        Config $config,
        StripeAdapterProvider $adapterProvider,
        EmailNotifier $emailNotifier,
        RepositoryInterface $subscriptionRepository,
        Date $date,
        RecurringTransactionGeneratorInterface $recurringTransactionGenerator,
        CompositeHandlerFactory $compositeHandlerFactory,
        HandleOrderContextFactory $handleOrderContextFactory,
        DateTime $dateTime
    ) {
        parent::__construct(
            $config,
            $emailNotifier,
            $subscriptionRepository,
            $date,
            $recurringTransactionGenerator
        );
        $this->adapterProvider = $adapterProvider;
        $this->subscriptionRepository = $subscriptionRepository;
        $this->compositeHandlerFactory = $compositeHandlerFactory;
        $this->handleOrderContextFactory = $handleOrderContextFactory;
        $this->dateTime = $dateTime;
    }

    /**
     * @param \Stripe\Event $event
     */
    public function process(\Stripe\Event $event)
    {
        $recurringTransaction = $this->saveTransactionLog($event, Status::SUCCESS);
        /** @var \Stripe\Invoice $object */
        $object = $event->data->object;
        if (!$object->amount_paid || !$object->charge) {
            return;
        }

        $subscription = $this->getSubscription($event);
        if (!$subscription) {
            return;
        }

        $transactionId = $object->charge ?? $object->id;

        /** @var HandleOrderContext $handleOrderContext */
        $handleOrderContext = $this->handleOrderContextFactory->create();

        $handleOrderContext->setSubscription($subscription);
        $handleOrderContext->setTransactionId($transactionId);
        $handleOrderContext->setRecurringTransaction($recurringTransaction);

        /** @var CompositeHandler $compositeHandler */
        $compositeHandler = $this->compositeHandlerFactory->create();

        $compositeHandler->handle($handleOrderContext);

        $subscription->setLastPaymentDate($this->dateTime->gmtDate());
        $this->subscriptionRepository->save($subscription);

        if ($subscription->getRemainingDiscountCycles() > 0) {
            $subscription->setRemainingDiscountCycles(
                $subscription->getRemainingDiscountCycles() - 1
            );
            $this->subscriptionRepository->save($subscription);

            if ($subscription->getRemainingDiscountCycles() === 0) {
                try {
                    $adapter = $this->adapterProvider->get();
                    $adapter->discountDelete($subscription->getSubscriptionId());
                } catch (\Exception $e) {
                    ;// do nothing
                }
            }
        }
    }
}
