<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2: Stripe (System)
 */

namespace Amasty\RecurringStripe\Controller\Stripe;

use Amasty\RecurringStripe\Api\IpnInterface;
use Amasty\RecurringStripe\Model\ConfigWebhook;
use Amasty\RecurringStripe\Model\StripeAdapterProvider;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Data\Form\FormKey;
use Psr\Log\LoggerInterface;

class Index implements HttpPostActionInterface, CsrfAwareActionInterface
{
    /**
     * @var StripeAdapterProvider
     */
    private $adapterProvider;

    /**
     * @var ConfigWebhook
     */
    private $configWebhook;

    /**
     * @var IpnInterface
     */
    private $stripeIpn;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * @var ResultFactory
     */
    private $resultFactory;

    public function __construct(
        ResultFactory $resultFactory,
        RequestInterface $request,
        StripeAdapterProvider $adapterProvider,
        ConfigWebhook $configWebhook,
        IpnInterface $stripeIpn,
        LoggerInterface $logger
    ) {
        $this->adapterProvider = $adapterProvider;
        $this->configWebhook = $configWebhook;
        $this->stripeIpn = $stripeIpn;
        $this->logger = $logger;
        $this->resultFactory = $resultFactory;
        $this->request = $request;
    }

    /**
     * @inheritDoc
     */
    public function execute()
    {
        /** @var \Magento\Framework\Controller\Result\Raw $result */
        $result = $this->resultFactory->create(ResultFactory::TYPE_RAW);
        /** @var RequestInterface $request */
        $request = $this->request;

        $payload = $request->getContent();
        $sigHeader = $request->getServer('HTTP_STRIPE_SIGNATURE');
        $event = null;

        try {
            $adapter = $this->adapterProvider->get();
            /** @var \Stripe\Event $event */
            $event = $adapter->eventRetrieve(
                (string)$payload,
                (string)$sigHeader,
                $this->configWebhook->getWebhookSecret()
            );
            $this->stripeIpn->processIpnRequest($event);
            $result->setHttpResponseCode(200);
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
            $result->setHttpResponseCode(400);
        }
        $result->setContents(''); // Prevent fatal error on Magento 2.3.3

        return $result;
    }

    /**
     * @param RequestInterface $request
     * @return InvalidRequestException|null
     */
    public function createCsrfValidationException(RequestInterface $request): ?InvalidRequestException
    {
        return null;
    }

    /**
     * @param RequestInterface $request
     * @return bool|null
     */
    public function validateForCsrf(RequestInterface $request): ?bool
    {
        return true;
    }
}
