<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2: Paypal (System)
 */

namespace Amasty\RecurringPaypal\Model\Subscription\Create;

use Amasty\RecurringPayments\Api\Subscription\SubscriptionInterface;
use Amasty\RecurringPayments\Model\Subscription\Create\CreateSubscriptionHandlerInterface;
use Amasty\RecurringPaypal\Api\ProductRepositoryInterface;
use Amasty\RecurringPaypal\Model\Api\Adapter;
use Amasty\RecurringPaypal\Model\PaypalProduct;
use Amasty\RecurringPaypal\Model\Processor\CreatePlan;
use Amasty\RecurringPaypal\Model\Processor\CreateProduct;
use Amasty\RecurringPaypal\Model\Processor\CreateSubscription;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Model\Quote\Item\AbstractItem;
use Magento\Sales\Api\Data\OrderInterface;
use PayPal\Exception\PayPalConnectionException;

class CreateSubscriptionHandler implements CreateSubscriptionHandlerInterface
{
    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * @var CreateProduct
     */
    private $createProduct;

    /**
     * @var CreatePlan
     */
    private $createPlan;

    /**
     * @var CreateSubscription
     */
    private $createSubscription;

    /**
     * @var Adapter
     */
    private $adapter;

    public function __construct(
        ProductRepositoryInterface $productRepository,
        CreateProduct $createProduct,
        CreatePlan $createPlan,
        CreateSubscription $createSubscription,
        Adapter $adapter
    ) {
        $this->productRepository = $productRepository;
        $this->createProduct = $createProduct;
        $this->createPlan = $createPlan;
        $this->createSubscription = $createSubscription;
        $this->adapter = $adapter;
    }

    /**
     * @param OrderInterface $order
     * @param AbstractItem $item
     * @param SubscriptionInterface $subscription
     */
    public function handle(OrderInterface $order, AbstractItem $item, SubscriptionInterface $subscription): void
    {
        $productId = $item->getProduct()->getId();

        try {
            /** @var PaypalProduct $product */
            $product = $this->productRepository->getByProductId($productId);
            // Ensure that product also exists on Paypal side
            $this->adapter->getProductDetails($product->getPaypalProductId());
        } catch (NoSuchEntityException $exception) {
            $product = $this->createProduct->execute($item, (int)$productId);
        } catch (PayPalConnectionException $exception) {
            $this->productRepository->delete($product);
            $product = $this->createProduct->execute($item, (int)$productId);
        }

        $planData = $this->createPlan->execute(
            $subscription,
            $item,
            (string)$product->getPaypalProductId(),
            $order
        );

        $subscriptionId = $this->createSubscription->execute(
            $subscription,
            $planData['id'],
            $order
        );
        $subscription->setSubscriptionId($subscriptionId);
    }
}
