<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2: Paypal (System)
 */

namespace Amasty\RecurringPaypal\Model\Subscription;

use Magento\Framework\App\CacheInterface;

class Cache
{
    public const TYPE_IDENTIFIER = 'amasty_recurring';
    public const CACHE_TAG = 'amasty_recurring';
    public const SUBSCRIPTION_ENTITY_KEY = 'subscription';
    public const BROKEN_RECORD = 'broken';
    public const LIFETIME = 3600 * 24;

    /**
     * @var CacheInterface
     */
    private $cache;

    public function __construct(
        CacheInterface $cache
    ) {
        $this->cache = $cache;
    }

    public function saveSubscriptionData(array $data): bool
    {
        unset($data['subscriber']); // Clear personal data

        return $this->cache->save(
            json_encode($data),
            $this->getSubscriptionKey($data['id']),
            [self::CACHE_TAG],
            self::LIFETIME
        );
    }

    public function clearSubscriptionData(string $subscriptionId): bool
    {
        return $this->cache->remove($this->getSubscriptionKey($subscriptionId));
    }

    /**
     * @param string $subscriptionId
     * @return array|bool|string
     */
    public function getSubscriptionData(string $subscriptionId)
    {
        $data = $this->cache->load($this->getSubscriptionKey($subscriptionId));
        if ($data && $data != self::BROKEN_RECORD) {
            return json_decode($data, true);
        }

        return $data;
    }

    public function markAsBroken(string $subscriptionId)
    {
        return $this->cache->save(self::BROKEN_RECORD, $this->getSubscriptionKey($subscriptionId));
    }

    protected function getSubscriptionKey(string $subscriptionId)
    {
        return self::TYPE_IDENTIFIER . '_' . self::SUBSCRIPTION_ENTITY_KEY . '_' . $subscriptionId;
    }
}
