<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2: Paypal (System)
 */

namespace Amasty\RecurringPaypal\Model;

use Amasty\RecurringPayments\Model\Config;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Encryption\Encryptor;

class ConfigProvider extends Config
{
    public const WEBHOOK_ID_XML_PATH = 'amasty_recurring_payments/paypal/webhook_id';
    public const EMAIL_TEMPLATE_SUBSCRIPTION_RENEWAL = 'email_template_paypal_renewal';
    public const SANDBOX_FLAG_PATH = 'paypal/wpp/sandbox_flag';

    public const LIVE_MODE = 'LIVE';
    public const SANDBOX_MODE = 'SANDBOX';

    /**
     * @var Encryptor
     */
    private $encryptor;

    public function __construct(
        ScopeConfigInterface $scopeConfig,
        Encryptor $encryptor
    ) {
        parent::__construct($scopeConfig);
        $this->encryptor = $encryptor;
    }

    /**
     * Returns pair of keys if both are filled and null otherwise
     * @return array|null
     */
    public function getPaypalCredentials()
    {
        $clientId = $this->getValue('paypal/client_id');
        $clientSecret = $this->encryptor->decrypt($this->getValue('paypal/client_secret'));

        if (empty($clientId) || empty($clientSecret)) {
            return null;
        }

        return [$clientId, $clientSecret];
    }

    /**
     * @return string
     */
    public function getPaypalWebhookId(): string
    {
        return (string)$this->getValue('paypal/webhook_id');
    }

    /**
     * @param int $storeId
     * @return string
     */
    public function getRenewalEmailTemplate(int $storeId): string
    {
        return (string)$this->getValue(
            self::EMAIL_NOTIFICATION_BLOCK . self::EMAIL_TEMPLATE_SUBSCRIPTION_RENEWAL,
            $storeId
        );
    }

    /**
     * @return string
     */
    public function getPaymentMode()
    {
        return $this->scopeConfig->getValue(self::SANDBOX_FLAG_PATH) ? self::SANDBOX_MODE : self::LIVE_MODE;
    }
}
