<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Push Notifications for Magento 2
 */

namespace Amasty\PushNotifications\Model;

use Amasty\PushNotifications\Api\Data\SubscriberInterface;
use Amasty\PushNotifications\Api\SubscriberRepositoryInterface;
use Amasty\PushNotifications\Exception\NotificationException;
use Amasty\PushNotifications\Model\ResourceModel\Subscriber as SubscriberResource;
use Amasty\PushNotifications\Model\ResourceModel\Subscriber\Collection;
use Amasty\PushNotifications\Model\ResourceModel\Subscriber\CollectionFactory;
use Magento\Framework\Api\Search\FilterGroup;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResultsInterface;
use Magento\Framework\Api\SortOrder;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Ui\Api\Data\BookmarkSearchResultsInterfaceFactory;

class SubscriberRepository implements SubscriberRepositoryInterface
{
    /**
     * @var BookmarkSearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    /**
     * @var SubscriberFactory
     */
    private $subscriberFactory;

    /**
     * @var SubscriberResource
     */
    private $subscriberResource;

    /**
     * Model data storage
     *
     * @var array
     */
    private $subscriber;

    /**
     * @var CollectionFactory
     */
    private $subscriberCollectionFactory;

    public function __construct(
        BookmarkSearchResultsInterfaceFactory $searchResultsFactory,
        SubscriberFactory $subscriberFactory,
        SubscriberResource $subscriberResource,
        CollectionFactory $subscriberCollectionFactory
    ) {
        $this->searchResultsFactory = $searchResultsFactory;
        $this->subscriberFactory = $subscriberFactory;
        $this->subscriberResource = $subscriberResource;
        $this->subscriberCollectionFactory = $subscriberCollectionFactory;
    }

    public function save(SubscriberInterface $subscriber): SubscriberInterface
    {
        try {
            $subscriber = $this->prepareSubscriberForSave($subscriber);

            $this->subscriberResource->save($subscriber);
            unset($this->subscriber[$subscriber->getSubscriberId()]);
        } catch (\Exception $e) {
            if ($subscriber->getSubscriberId()) {
                throw new CouldNotSaveException(
                    __(
                        'Unable to save subscriber with ID %1. Error: %2',
                        [$subscriber->getSubscriberId(), $e->getMessage()]
                    )
                );
            }
            throw new CouldNotSaveException(__('Unable to save new subscriber. Error: %1', $e->getMessage()));
        }

        return $subscriber;
    }

    public function getById(int $subscriberId): SubscriberInterface
    {
        if (!isset($this->subscriber[$subscriberId])) {
            return $this->getSubscriberByField($subscriberId);
        }

        return $this->subscriber[$subscriberId];
    }

    public function getByToken(string $token): SubscriberInterface
    {
        return $this->getSubscriberByField($token, SubscriberInterface::TOKEN);
    }

    public function getByCustomerVisitor(?string $customerId, ?string $visitorId)
    {
        if ($customerId) {
            return $this->getSubscriberByField($customerId, SubscriberInterface::CUSTOMER_ID);
        } elseif ($visitorId) {
            return $this->getSubscriberByField($visitorId, SubscriberInterface::VISITOR_ID);
        }

        throw new NotificationException(__('Customer Id or Visitor Id is not defined'));
    }

    /**
     * @param string|int $fieldValue
     * @param string $fieldName
     *
     * @return Subscriber|bool
     *
     * @throws NotificationException
     */
    private function getSubscriberByField(
        $fieldValue,
        string $fieldName = SubscriberInterface::SUBSCRIBER_ID
    ) {
        if ($fieldValue) {
            $subscriber = $this->subscriberFactory->create();
            $this->subscriberResource->load($subscriber, $fieldValue, $fieldName);

            if (!$subscriber->getSubscriberId()) {
                return false;
            }

            $this->subscriber[$subscriber->getSubscriberId()] = $subscriber;

            return $subscriber;
        }

        throw new NotificationException(__('Field value is not defined.'));
    }

    public function delete(SubscriberInterface $subscriber): bool
    {
        try {
            $this->subscriberResource->delete($subscriber);
            unset($this->subscriber[$subscriber->getSubscriberId()]);
        } catch (\Exception $e) {
            if ($subscriber->getSubscriberId()) {
                throw new CouldNotDeleteException(
                    __(
                        'Unable to remove question with ID %1. Error: %2',
                        [$subscriber->getSubscriberId(), $e->getMessage()]
                    )
                );
            }

            throw new CouldNotDeleteException(__('Unable to remove subscriber. Error: %1', $e->getMessage()));
        }

        return true;
    }

    public function deleteById(int $subscriberId): bool
    {
        $subscriberModel = $this->getById($subscriberId);
        if ($subscriberModel) {
            $this->delete($subscriberModel);
        }

        return true;
    }

    public function deleteByCustomerId(int $customerId): bool
    {
        if ($subscriberModel = $this->getByCustomerVisitor($customerId, null)) {
            $this->delete($subscriberModel);
        }

        return true;
    }

    public function getList(SearchCriteriaInterface $searchCriteria): SearchResultsInterface
    {
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $subscriberCollection = $this->subscriberCollectionFactory->create();

        foreach ($searchCriteria->getFilterGroups() as $group) {
            $this->addFilterGroupToCollection($group, $subscriberCollection);
        }

        $searchResults->setTotalCount($subscriberCollection->getSize());
        $sortOrders = $searchCriteria->getSortOrders();

        if ($sortOrders) {
            $this->addOrderToCollection($sortOrders, $subscriberCollection);
        }

        $subscriberCollection->setCurPage($searchCriteria->getCurrentPage());
        $subscriberCollection->setPageSize($searchCriteria->getPageSize());
        $subscribers = [];

        /** @var SubscriberInterface $subscriber */
        foreach ($subscriberCollection->getItems() as $subscriber) {
            $subscribers[] = $this->getById($subscriber->getSubscriberId());
        }

        $searchResults->setItems($subscribers);

        return $searchResults;
    }

    /**
     * Helper function that adds a FilterGroup to the collection.
     */
    private function addFilterGroupToCollection(FilterGroup $filterGroup, Collection $subscriberCollection): void
    {
        foreach ($filterGroup->getFilters() as $filter) {
            $condition = $filter->getConditionType() ? $filter->getConditionType() : 'eq';
            $subscriberCollection->addFieldToFilter(
                $filter->getField(),
                [
                    $condition => $filter->getValue()
                ]
            );
        }
    }

    /**
     * Helper function that adds a SortOrder to the collection.
     */
    private function addOrderToCollection(array $sortOrders, Collection $subscriberCollection): void
    {
        /** @var SortOrder $sortOrder */
        foreach ($sortOrders as $sortOrder) {
            $field = $sortOrder->getField();
            $subscriberCollection->addOrder(
                $field,
                ($sortOrder->getDirection() == SortOrder::SORT_DESC) ? 'DESC' : 'ASC'
            );
        }
    }

    /**
     * @param SubscriberInterface $subscriber
     *
     * @return SubscriberInterface|mixed
     *
     * @throws NoSuchEntityException
     */
    private function prepareSubscriberForSave(SubscriberInterface $subscriber)
    {
        if ($subscriber->getSubscriberId()) {
            $savedSubscriber = $this->getById($subscriber->getSubscriberId());
            $savedSubscriber->addData($subscriber->getData());

            return $savedSubscriber;
        }

        return $subscriber;
    }
}
