<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Push Notifications for Magento 2
 */

namespace Amasty\PushNotifications\Api;

use Amasty\PushNotifications\Api\Data\SubscriberInterface;
use Amasty\PushNotifications\Exception\NotificationException;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResultsInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

/**
 * @api
 */
interface SubscriberRepositoryInterface
{
    /**
     * Save
     *
     * @param SubscriberInterface $subscriber
     * @return SubscriberInterface
     * @throws CouldNotSaveException
     */
    public function save(SubscriberInterface $subscriber): SubscriberInterface;

    /**
     * Get by id
     *
     * @param int $subscriberId
     * @return SubscriberInterface
     * @throws NotificationException
     */
    public function getById(int $subscriberId): SubscriberInterface;

    /**
     * Get by token
     *
     * @param string $token
     * @return SubscriberInterface
     * @throws NotificationException
     */
    public function getByToken(string $token): SubscriberInterface;

    /**
     * Get by customer id or visitor id
     *
     * @param string|null $customerId
     * @param string $visitorId
     * @return SubscriberInterface|bool
     * @throws NotificationException
     */
    public function getByCustomerVisitor(?string $customerId, string $visitorId);

    /**
     * Delete
     *
     * @param SubscriberInterface $subscriber
     * @return bool true on success
     * @throws CouldNotDeleteException
     */
    public function delete(SubscriberInterface $subscriber): bool;

    /**
     * Delete by subscriber id
     *
     * @param int $subscriberId
     * @return bool true on success
     * @throws CouldNotDeleteException
     */
    public function deleteById(int $subscriberId): bool;

    /**
     * Delete by customer id
     *
     * @param int $customerId
     * @return bool true on success
     * @throws CouldNotDeleteException
     */
    public function deleteByCustomerId(int $customerId): bool;

    /**
     * Lists
     *
     * @param \Magento\Framework\Api\SearchCriteriaInterface $searchCriteria
     * @return \Magento\Framework\Api\SearchResultsInterface
     * @throws NoSuchEntityException
     */
    public function getList(SearchCriteriaInterface $searchCriteria): SearchResultsInterface;
}
