<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Countdown Timer for Magento 2
 */

namespace Amasty\PromoCountdown\UI\DataProvider;

use Amasty\Base\Model\Serializer;
use Amasty\PromoCountdown\Block\Widgets\Countdown;
use Amasty\PromoCountdown\Model\OptionSource\Status;
use Magento\Framework\Stdlib\DateTime\DateTime;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Ui\DataProvider\AbstractDataProvider;
use Magento\Widget\Model\ResourceModel\Widget\Instance\Collection;
use Magento\Widget\Model\ResourceModel\Widget\Instance\CollectionFactory;

class ListingDataProvider extends AbstractDataProvider
{
    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var DateTime
     */
    private $dateTime;

    /**
     * @var TimezoneInterface
     */
    private $timezone;

    public function __construct(
        $name,
        $primaryFieldName,
        $requestFieldName,
        CollectionFactory $collectionFactory,
        Serializer $serializer,
        DateTime $dateTime,
        TimezoneInterface $timezone,
        array $meta = [],
        array $data = []
    ) {
        parent::__construct($name, $primaryFieldName, $requestFieldName, $meta, $data);

        $this->collection = null;
        $this->collectionFactory = $collectionFactory;
        $this->serializer = $serializer;
        $this->dateTime = $dateTime;
        $this->timezone = $timezone;
    }

    /**
     * @return Collection
     */
    public function getCollection()
    {
        if (!$this->collection) {
            $this->collection = $this->collectionFactory->create()
                ->addFieldToFilter('instance_type', Countdown::class);
        }

        return parent::getCollection();
    }

    /**
     * @return array
     */
    public function getData()
    {
        $collection = $this->getCollection();
        $data['totalRecords'] = $collection->getSize();
        $data['items'] = [];

        /** @var \Magento\Widget\Model\Widget\Instance $item */
        foreach ($collection->getItems() as $item) {
            $params = $item->getWidgetParameters();

            if (!$params) {
                $params = $this->serializer->unserialize($item->getData('widget_parameters'));
            }

            $itemData = $item->getData();
            $itemData['date_to'] = empty($params['date_to']) ? __('No end time.') : $params['date_to'];
            $itemData['date_from'] = empty($params['date_from']) ? __('No start time.') : $params['date_from'];
            $itemData['status'] = $this->getStatus($params);
            $data['items'][] = $itemData;
        }

        return $data;
    }

    /**
     * @param array $params
     * @return int
     */
    private function getStatus(array $params): int
    {
        $date = $this->timezone->scopeTimeStamp();
        $dateFrom = $this->dateTime->gmtTimestamp($params['date_from']);
        $dateTo = $this->dateTime->gmtTimestamp($params['date_to']);

        $result = ($date > $dateTo) ? Status::EXPIRED : Status::STOPPED;

        if (isset($params['am_countdown_active']) && $params['am_countdown_active']) {
            if ($date < $dateFrom) {
                $result = Status::SCHEDULED;
            } else {
                $result = ($date < $dateTo) ? Status::ACTIVE : Status::EXPIRED;
            }
        }

        return $result;
    }
}
